<h2>Why is this an issue?</h2>
<p>In TypeScript, there are two ways to define properties or parameters that are potentially <code>undefined</code>:</p>
<ul>
  <li> Union with <code>undefined</code>: Adding <code>| undefined</code> in the property type makes the property <em>required</em>, but can be
  <code>undefined</code>. Use this syntax when you want to be explicit that an object should provide that property, in which case the TypeScript
  compiler will not allow omitting it. </li>
</ul>

```ts
interface Person {
  name: string;
  address: string | undefined;
}

let John = { name: "John", address: undefined };
```
<ul>
  <li> Optional property syntax (<code>?</code> after its name): The property is <em>optional</em>, which means that an object can omit it and let the
  TypeScript compiler provide it as being <code>undefined</code>. </li>
</ul>

```ts
interface Person {
  name: string;
  address?: string;
}

let John = { name: "John" };
```
<p>This rule checks for optional property declarations that use both the <code>?</code> syntax and unions with <code>undefined</code>.</p>

```ts
interface Person {
  name: string;
  address?: string | undefined;   // Noncompliant: using both syntaxes is redundant
}
```
<p>Choose one of the syntaxes to declare optional properties and remove the other one. Consider using only <code>| undefined</code> if you want to
make the property explicit in the object.</p>

```ts
interface Person {
  name: string;
  address?: string;
}
```
<p>The rule does not raise any issues when the TypeScript compiler option <code>exactOptionalPropertyTypes</code> is enabled because this option
ensures that <code>undefined</code> does not become redundant in this context.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> TypeScript Documentation - <a href="https://www.typescriptlang.org/docs/handbook/2/objects.html#optional-properties">Optional Properties</a>
  </li>
  <li> TypeScript Documentation - <a href="https://www.typescriptlang.org/docs/handbook/2/everyday-types.html#union-types">Union Types</a> </li>
  <li> TypeScript Documentation - <a href="https://www.typescriptlang.org/tsconfig/#exactOptionalPropertyTypes">exactOptionalPropertyTypes</a> </li>
</ul>