<h2>Why is this an issue?</h2>
<p>TypeScript allows all sorts of expressions to initialize enum members. However, as enums create their own scope, using identifiers can lead to
unexpected results. The recommendation is thus to use only literal values when defining enum members.</p>
<h3>What is the potential impact?</h3>
<p>An enum member will shadow any variable defined with the same name in an enclosing scope. In the code example below, the <code>day</code> member
value should derive from the <code>hour</code> const variable. But it is computed using the enum <code>hour</code> member instead, leading to the
result 24 instead of the expected 1440.</p>

```ts
const hour = 3600; // hour in seconds

enum DurationInMinutes {
    hour = 60, // hour in minutes
    day = 24 * hour / 60 // Expecting hour in seconds but got hour in minutes
}
```
<h2>How to fix it</h2>
<p>The recommended approach for enums is to initialize members with literals. Not only are literals easier to read, but they also avoid unexpected
results due to the enum scope.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```ts
const hello = 'Hello';
enum Foo {
  STRING = hello, // Variable
  OBJECT = { hello }.hello.length, // Object
  TEMPLATE = `${hello}, World`, // Template literal
  SET = new Set([hello, 'world']).size, // Constructor
  NUMBER = hello.length + 1, // Expression
}
```
<h4>Compliant solution</h4>

```ts
enum Foo {
  STRING = 'Hello',
  NUMBER = 0,
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> TypeScript Documentation - <a href="https://www.typescriptlang.org/docs/handbook/enums.html#handbook-content">Enums</a> </li>
</ul>