<h2>Why is this an issue?</h2>
<p>React does not always propagate component state changes to the underlying DOM elements immediately. If you modify the state during an already
ongoing update cycle, the change may be delayed until the next update. React tries to keep <code>this.state</code> in sync with what is currently in
the DOM, so you should never directly modify <code>this.state</code> yourself. Instead, use the asynchronous <code>setState</code> method instead,
which allows React to properly manage the current state, trigger the new update cycle, or batch the updates together if necessary.</p>
<p>Note that <code>setState()</code> is a <em>request</em> to change the state, not an immediate update. For example, if multiple components are
changing the state in response to a user event, React will wait until the event handler has finished executing and then render all the changes in a
single update. In other cases, the updates may be executed one at a time, so you should never make assumptions about how the component state will
change in response to your request.</p>
<p>If your next state is a function of the current state, you should pass an updater function to the <code>setState()</code> that will give you access
to the correct component state at the time of the execution.</p>
<p>The only place where you should directly modify the state is during the component initialization in a constructor function.</p>

```ts
class MyComponent extends React.Component {
  constructor() {
    super();
    this.state = {
      count: 0
    };
  }

  incrementCount() {
    this.state.count++; // Noncompliant: direct mutation of state object
  }

  render() {
    return (
      <div>
        <p>Count: {this.state.count}</p>
        <button onClick={this.incrementCount}>Increment</button>
      </div>
    );
  }
}
```
<p>To fix this code use <code>setState()</code> method instead of directly mutating the state.</p>

```ts
class MyComponent extends React.Component {
  constructor() {
    super();
    this.state = {
      count: 0
    };
  }

  incrementCount() {
    this.setState(prevState => ({
      count: prevState.count + 1
    }));
  }

  render() {
    return (
      <div>
        <p>Count: {this.state.count}</p>
        <button onClick={this.incrementCount}>Increment</button>
      </div>
    );
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a href="https://react.dev/reference/react/Component#setstate"><code>setState()</code> method</a> </li>
  <li> React Documentation - <a href="https://react.dev/learn/managing-state">Managing state</a> </li>
</ul>