Deeply nested functions are more difficult to understand and maintain, as well as, test.
Extract helper functions or apply an early-return refactoring to reduce nesting.

# What Does This Check Look For?

This check counts the nesting depth, starting with 0 at the function or method level.
Each control structure, such as conditionals or loops, adds one nesting level.
Depending on the analysis profile configuration, this check produces yellow findings for slightly nested functions (default: nesting depth of more than 3) and red findings for deeply nested functions (default: nesting depth of more than 5).

# Why is This a Problem?

Deep nesting is an issue for two related, but distinct reasons:

1. It complicates the comprehensibility of the control flow and makes maintenance more difficult.
It requires the programmer to hold a lot of information in their head to understand the conditions holding at a particular location in the control flow.
Additionally, the end of each nested construct is often hard to spot.

2. Deeply nested functions are more difficult to test exhaustively, because more paths have to be covered.

# How Can I Resolve This?

To resolve deep nesting, you can extract parts of the code into separate helper functions.

Often, you may also reduce nesting by merging conditional statements or rephrasing positive conditionals to negative ones in combination with a return statement or a `continue` statement in a loop.
This is also known as early return, early exit or guard clause refactoring.
Note: While there is sometimes a notion of having 'one return only', this is no longer deemed a best practice that should be followed (see also references).

# Examples
## Non-Compliant

This example uses Java, but the principle generalizes to other languages.

```java
public static boolean isListContainedInList(List<Object> container, List<Object> searchedList) {
    if (container != null && container.size() != 0) {
        if (searchedList != null) {
             if (container.size() >= searchedList.size()) {
                  for (int i = 0; i <= container.size() - searchedList.size(); i++) {
                      boolean result = true;
                      for (int startIndex = 0; startIndex < searchedList.size(); startIndex++) {
                           if (!container.get(i + startIndex).equals(searchedList.get(startIndex))) {
                                result = false;
                                break;
                           }
                      }

                      if (result) {
                           return true;
                      }
                  }
             } else {
                  return false;
             }
        } else {
             // it is very difficult to see which condition this return belongs to
             return true;
        }
    } else {
        return false;
    }

    return false;
}
```

## Compliant

Note how the multiple early returns in the beginning of the method make the error cases more explicit and easier to read.

```java
public static boolean isListContainedInList(List<Object> container, List<Object> searchedList) {
    if (container == null || container.size() == 0) {
        // early return
        return false;
    }

    if (searchedList == null) {
        return true;
    }

    if (container.size() < searchedList.size()) {
        return false;
    }

    for (int i = 0; i <= container.size() - searchedList.size(); i++) {
        if (isListContainedStartingAt(container, searchedList, i)) {
             return true;
        }
    }

    return false;
}

// helper function with speaking name
private static boolean isListContainedStartingAt(List<Object> container, List<Object> searchedList, int startIndex) {
    for (int i = 0; i < searchedList.size(); i++) {
        if (!container.get(startIndex + i).equals(searchedList.get(i))) {
             return false;
        }
    }

    return true;
}
```

# Where Can I Learn More?

* [Teamscale Documentation: What Makes Maintainable Code? - Code Structure](https://docs.teamscale.com/introduction/understanding-the-maintainability-of-your-code-base/#code-structure)
* [Refactoring: Replace Nested Conditional with Guard Clauses](https://refactoring.com/catalog/replaceNestedConditionalWithGuardClauses.html)
* [Stackoverflow: Where did the notion of “one return only” come from?](https://softwareengineering.stackexchange.com/questions/118703/where-did-the-notion-of-one-return-only-come-from)
* Clean Code: A Handbook of Agile Software Craftsmanship. Chapter 3: Functions (Martin, 2008)
