The code contains critical sections that overlap each other.

# What Does This Check Look For?

This check identifies areas in the code where critical sections overlap, which could involve locking methods or synchronized blocks/methods.

# Why is This a Problem?

Critical sections are used to synchronize access to shared resources from multiple threads.
Overlapping critical sections in general and nested critical sections in particular can be defined by using different lock entities (this includes monitor locks, i.e., `synchronized` blocks or methods).
There are two potential issues when using overlapping critical regions:

- Deadlocks happen when two threads are waiting for unlocking the lock held by the other thread.
  In these situations, the respective tasks make no progress anymore.
- Overlapping critical regions improve the complexity of the code and the runtime behavior.
  Developers have to consider every possible lock combination of the involved locks.

# How Can I Resolve This?

Split the tasks of critical sections into multiple sub-tasks that communicate, e.g., via concurrent queues.

# Examples

## Non-Compliant

```java
public class OverlappingCriticalSections {

  private final Object lock1 = new Object();
  private final Object lock2 = new Object();
  private final Lock lock3 = new ReentrantLock();
  private final Lock lock4 = new ReentrantLock();

  public synchronized void criticalMethod1() {
    System.out.println("Critical Section 1 - Method");
    synchronized (lock2) {
      System.out.println("Nested Lock on lock2 in criticalMethod1");
    }
  }

  public void criticalMethod2() {
    synchronized (lock2) {
      System.out.println("Critical Section 2 - lock2 in criticalMethod2");
      synchronized (lock1) {
        System.out.println("Nested Lock on lock1 in criticalMethod2");
      }
    }
  }

  public void criticalMethod3() {
    lock3.lock();
    lock4.lock();
    try {
      // access the resource protected by this lock
    } finally {
      lock4.unlock();
      lock3.unlock();
    }
  }
}
```

## Compliant

Note that fixing this requires use-case specific information so these fixes here are just exemplary.

```java
public class NonOverlappingCriticalSections {

  private final Object lock1 = new Object();
  private final Object lock2 = new Object();
  private final Lock lock3 = new ReentrantLock();
  private final Lock lock4 = new ReentrantLock();

  public synchronized void criticalMethod1() {
    System.out.println("Critical Section 1 - Method");
  }

  public void criticalMethod2() {
    synchronized (lock2) {
      System.out.println("Critical Section 2 - lock2 in criticalMethod2");
    }
    synchronized (lock1) {
      System.out.println("Lock on lock1 in criticalMethod2");
    }
  }

  public void criticalMethod3() {
    lock3.lock();
    try {
      // access the resource protected by this lock
    } finally {
      lock3.unlock();
    }
  }
}
```

# Where Can I Learn More?

- [Code Review Checklist (Java Concurrency): Item DI.1](https://github.com/code-review-checklists/java-concurrency#avoid-nested-critical-sections)
- [Code Review Checklist (Java Concurrency): Item DI.2](https://github.com/code-review-checklists/java-concurrency#document-locking-order)
- [Code Review Checklist (Java Concurrency): Item DI.3](https://github.com/code-review-checklists/java-concurrency#dynamic-lock-ordering)
