This check detects `SELECT` statements that use `BULK COLLECT` and `FETCH` statements that use `BULK COLLECT` without a `LIMIT` clause.

# Why is This a Problem?
When using `BULK COLLECT` each fetch of a database table returns multiple rows.
While this can reduce execution time, fetching multiple rows can cause serious problems since it uses more memory.
For large tables, this can lead to memory exhaustion.

Since `SELECT` statements are implicit queries, both `SELECT BULK COLLECT` statements and `FETCH BULK COLLECT` statements can cause memory exhaustion.

# How Can I Resolve This?
`FETCH BULK COLLECT` statements should always be used in combination with a `LIMIT` clause and a `LOOP` to process the received data in chunks.

`SELECT BULK COLLECT` statements should be replaced with `FETCH BULK COLLECT` statements with a `LIMIT` clause to avoid memory exhaustion due to the implicit multi-row fetch.

## Example (Before)

```oraclesqlplus
BEGIN
    OPEN foo;
    FETCH foo BULK COLLECT INTO bar;
END;

BEGIN
    SELECT * BULK COLLECT INTO bar FROM someTable;
END;

```

## Example (After)

```oraclesqlplus
BEGIN
    OPEN foo;

    LOOP
        FETCH foo BULK COLLECT INTO bar LIMIT 100;
        EXIT WHEN foo%NOTFOUND; 
    END LOOP;
END;

DECLARE
    CURSOR cursor is SELECT * FROM someTable;
BEGIN
    OPEN cursor;

    LOOP
        FETCH cursor BULK COLLECT INTO bar LIMIT 100;
        EXIT WHEN bar.COUNT = 0;
    END LOOP;
END;
```

# Where Can I Learn More?
## References
- [Obsessed with Oracle PL/SQL: How to Pick the Limit for BULK COLLECT](http://stevenfeuersteinonplsql.blogspot.com/2018/10/how-to-pick-bulk-collect-limit.html)
