Catching generic exceptions can mask the root cause of an error, making debugging and error handling more difficult.
Prefer catching specific exceptions to handle known error conditions safely and clearly.

# What Does This Check Look For?

This check flags catch blocks that handle overly generic exceptions, such as `Throwable`, `Error`, `RuntimeException`, and `Exception` in Java or `Exception`, `ApplicationException`, and `SystemException` in C#.

# Why is This a Problem?

Catching generic exceptions can lead to several problems:
- It can make error diagnosis and recovery more difficult, as the catch block may handle a wide range of errors, including those not anticipated by the developer.
- It potentially masks more specific errors that could have been handled individually or reported more clearly.
- It violates the principle of catching exceptions at the level where they can be handled meaningfully.

# How Can I Resolve This?

To resolve this issue, consider catching more specific exceptions that you expect in the context. This approach allows for more granular error handling and recovery:
- Review the code and documentation to understand which exceptions might be thrown.
- Replace generic exception catches with specific ones where possible.
- If a generic catch is essential, rethrow the exception after logging or performing necessary cleanup actions.

# Examples

## Non-Compliant

```java
void calculate(int possiblyZero) {
	try {
		System.out.println(1000 / possiblyZero);
	} catch (Exception e) {
		e.printStackTrace();
	}
}
```

## Compliant

```java
void calculate(int possiblyZero) {
	try {
		System.out.println(1000 / possiblyZero);
	} catch (ArithmeticException e) {
		e.printStackTrace();
	}
}
```
