Calling constructors for encoding-sensitive objects without providing an encoding could lead to compatibility issues.
A default encoding should be specified instead.

# What Does This Check Look For?
This check detects constructors for encoding-sensitive objects that include no parameters to set an encoding, e.g. `String` constructor calls with the last parameter not being of type `String` or `Charset`.

# Why is This a Problem?
If no charset (i.e. no encoding) is provided to objects that interpret `Byte`s as text (e.g. `String` or `FileReader`), the compiler will use the default settings of the coding environment.
In this case, different machines may use different default encodings.
As a result, encoded strings and streams, for example, may differ unexpectedly and cause **unwanted behavior**.


# How Can I Resolve This?
You should **specify a charset** when calling constructors for encoding-senstive objects that handle `Byte` arrays or files.
Always prefer the use of standardized charsets for compatability and maintainability reasons, for example the **Java Standard Charset** (package: `java.nio.charset.StandardCharsets`).

This solution does not work for Java `FileReader` before Java 11 since specifying an encoding was not supported. 
In this case, one could use `new InputStreamReader(new FileInputStream(filePath), desiredEncoding)` to specify an encoding.

# Examples

## Non-Compliant

```java
public static void nonCompliantFunction(byte [] raw, InputStream inputStream, File file) {
    String s = new String(raw);     // non-compliant: no encoding used
    System.out.println(s);

    InputStreamReader streamReader = new InputStreamReader(inputStream);     // non-compliant: no encoding used

    FileReader fileReader = new FileReader(file);     // non-compliant: no encoding used
}
```

## Compliant

```java
public static void compliant(byte [] raw, InputStream inputStream, File file) {
    String s = new String(raw, StandardCharsets.UTF_8);     // compliant: using UTF-8 encoding as an example
    System.out.println(s);

    InputStreamReader streamReader = new InputStreamReader(inputStream, StandardCharsets.UTF_8);     // compliant: using UTF-8 encoding as an example

    FileReader fileReader = new FileReader(file, StandardCharsets.UTF_8); // compliant: correct initialization using UTF-8 for FileReader for Java 11+ 
}
```

# Where Can I Learn More?

* [The Java™ Tutorials: Byte Encodings and Strings](https://docs.oracle.com/javase/tutorial/i18n/text/string.html)
* [Javatpoint: Java String Encoding](https://www.javatpoint.com/java-string-encoding)

