Empty try/finally blocks can lead to confusion about whether they were intentionally left empty or were supposed to be implemented later. 
To avoid confusion, developers should either implement the necessary logic or provide an explanatory comment within the blocks.

# What Does This Check Look For?

This check identifies instances in the code where `try`/`finally` blocks contain no statements or comments, either in the `try` block, the `finally` block, or both.

# Why is This a Problem?

Empty `try`/`finally` blocks might indicate a misunderstanding or neglect of proper error handling mechanisms, which could lead to uncaught exceptions or missed resource management opportunities.
They also make the code harder to read and maintain, as it's unclear whether these blocks were intentionally left empty or are placeholders for future implementation.
**Clarification is essential** to maintain code quality and understandability.

# How Can I Resolve This?

Developers have several options to address this issue:
- If the blocks were intentionally left empty, add a comment within them explaining why no action is needed.
This keeps the code self-documenting and alleviates confusion.
- If the blocks should contain logic (error handling or resource management in the `finally` block), implement the necessary code.
This ensures that exceptions are handled correctly and resources are managed appropriately.

# Examples

## Non-Compliant

```java
try {
    FileReader fr = new FileReader("file.txt");
} finally {
    
}
```

## Compliant

```java
try {
    FileReader fr = new FileReader("file.txt");
} finally {
    // Closing resources
    resource.close();
}
```

Or, if intentionally left empty:

```java
try {
    FileReader fr = new FileReader("file.txt");
} finally {
    // intentionally left empty because...
}
```

# Where Can I Learn More?

- [Oracle Java Tutorials: The try Block](https://docs.oracle.com/javase/tutorial/essential/exceptions/try.html): Explains the importance of `try`/`finally` and how to use them correctly.
