Using double-precision math functions (e.g., `exp()`, `sin()`, `log()`) from `math.h` may increase the size of compiled binaries. Replacing these functions with their single-precision counterparts (e.g., `expf()`, `sinf()`, `logf()`) can reduce binary size and optimize resource usage.

# What Does This Check Look For?

This check detects the usage of double-precision math functions from `math.h` by matching their names (e.g., `exp`, `sin`, `log`). It does not analyze the context of the arguments or differentiate between necessary and unintentional double-precision usage. Additionally, this check does not produce findings for functions declared inside macros.

# Why is This a Problem?

Double-precision functions increase binary size due to additional computational requirements and dependencies. When single-precision accuracy suffices, using float versions can reduce binary size and improve performance.

# How Can I Resolve This?

- Review if double-precision is genuinely required for your use case.
- Replace double-precision functions with their single-precision equivalents (e.g., use `expf()` instead of `exp()`).

# Examples

## Non-Compliant
```c
#include <math.h>

void foo(){
   double distance = sqrt(25.0); // non-compliant: using double-based precision function
   //... 
}
```
## Compliant
```c
#include <math.h>

void foo(){
   float distance = sqrtf(25.0f); // comlpiant: using float-based precision function
   //... 
}
```
# Where Can I Learn More?
- [Common mathematical functions](https://en.cppreference.com/w/c/numeric/math)
- [Memory Layout of C Program](https://www.hackerearth.com/practice/notes/memory-layout-of-c-program/)