/*
 * Decompiled with CFR 0.152.
 */
package com.teamscale.index.system_info;

import com.teamscale.core.option.server.ServerOptionIndex;
import com.teamscale.core.options.RepositoryCloneOption;
import com.teamscale.core.user.UserSatisfactionFeedbackIndex;
import com.teamscale.index.monitoring.prometheus.SystemLoadMetricsCollector;
import com.teamscale.index.system_info.SystemInfoEntry;
import com.teamscale.index.system_info.SystemInfoFragmentBase;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.sshd.common.util.OsUtils;
import org.conqat.engine.persistence.store.StorageException;
import org.conqat.lib.commons.collections.CollectionUtils;
import org.conqat.lib.commons.collections.CounterSet;
import org.conqat.lib.commons.date.DateTimeUtils;
import org.conqat.lib.commons.filesystem.ByteUnit;
import org.conqat.lib.commons.filesystem.FileSystemUtils;
import org.conqat.lib.commons.test.IndexValueClass;
import org.jspecify.annotations.NonNull;
import org.jspecify.annotations.Nullable;
import oshi.SystemInfo;
import oshi.hardware.CentralProcessor;
import oshi.util.FileUtil;

@IndexValueClass
public class SystemLoadFragment
extends SystemInfoFragmentBase {
    private static final long serialVersionUID = 1L;
    private static final Logger LOGGER = LogManager.getLogger();
    private final String cpuDescription;
    private final double cpuCoreCount;
    private final double cpuLoad;
    private final long ramInstalledMB;
    private final long ramUsedMB;
    private final PathWithSpaceInfo workingDirectory;
    private final @Nullable PathWithSpaceInfo storageDirectory;
    private final long storageDirectorySize;
    private final PathWithSpaceInfo repositoriesDirectory;
    private final PathWithSpaceInfo tempDirectory;
    private final String systemTime;
    private final String userSatisfaction;
    private static final Pattern TWO_LONG_VALUES = Pattern.compile("^ *([0-9]+) +([0-9]+)\\s*$");

    public SystemLoadFragment(SystemInfo systemInfo, @Nullable File storageDirectory, ServerOptionIndex serverOptionIndex, UserSatisfactionFeedbackIndex userSatisfactionFeedbackIndex, long updateInterval, long[] previousTicks) throws StorageException, IOException {
        super(updateInterval);
        Optional<CentralProcessor> cpuInfo = SystemLoadFragment.obtainCpuInfo(systemInfo);
        if (cpuInfo.isPresent()) {
            this.cpuDescription = cpuInfo.get().getProcessorIdentifier().getName();
            this.cpuCoreCount = SystemLoadFragment.determineCpuCoreCount(cpuInfo.get());
            this.cpuLoad = SystemLoadFragment.determineCpuLoad(cpuInfo.get(), this.cpuCoreCount, previousTicks);
        } else {
            this.cpuDescription = "<CPU info cannot be retrieved. See logs.>";
            this.cpuCoreCount = 0.0;
            this.cpuLoad = 0.0;
        }
        this.ramInstalledMB = SystemLoadFragment.getRamInstalledMB(systemInfo);
        this.ramUsedMB = SystemLoadFragment.getRamUsedMB(systemInfo);
        this.workingDirectory = new PathWithSpaceInfo(FileSystemUtils.getJvmWorkingDirOrTempForDevMode());
        this.repositoriesDirectory = new PathWithSpaceInfo(RepositoryCloneOption.getRepositoryCloneDirectory((ServerOptionIndex)serverOptionIndex).toFile());
        this.tempDirectory = new PathWithSpaceInfo(FileSystemUtils.getTmpDir());
        if (storageDirectory != null) {
            this.storageDirectory = new PathWithSpaceInfo(storageDirectory);
            this.storageDirectorySize = SystemLoadFragment.obtainStorageDirectorySize(storageDirectory);
        } else {
            this.storageDirectory = null;
            this.storageDirectorySize = 0L;
        }
        this.systemTime = DateTimeUtils.UI_FORMATTER.format(LocalDateTime.now());
        this.userSatisfaction = this.addUserSatisfactionDetails(userSatisfactionFeedbackIndex);
        SystemLoadMetricsCollector.collect(this);
    }

    private static long obtainStorageDirectorySize(File storageDirectory) {
        return FileSystemUtils.calculateDirectorySize((Path)storageDirectory.toPath(), failedFiles -> LOGGER.debug("Failed to compute the size of some files in storage directory. Computed directory size may be incorrect. This is not an issue, since files may be moved around during the computation. Failed files are: {}", failedFiles));
    }

    private static Optional<CentralProcessor> obtainCpuInfo(SystemInfo systemInfo) {
        try {
            CentralProcessor cpuInfo = systemInfo.getHardware().getProcessor();
            return Optional.of(cpuInfo);
        }
        catch (Throwable t) {
            LOGGER.error("Error obtaining CPU information with OSHI library: {}", (Object)t, (Object)t);
            return Optional.empty();
        }
    }

    private String addUserSatisfactionDetails(UserSatisfactionFeedbackIndex userSatisfactionFeedbackIndex) throws StorageException {
        CounterSet ratings = new CounterSet();
        HashSet userSatisfactionFeedbacks = CollectionUtils.unionSetAll((Collection)userSatisfactionFeedbackIndex.getAllUserSatisfactionEntries().extractSecondList());
        for (UserSatisfactionFeedbackIndex.UserSatisfactionFeedback userSatisfactionFeedback : userSatisfactionFeedbacks) {
            ratings.inc((Object)userSatisfactionFeedback.rating());
        }
        Object userSatisfaction = "no data";
        if (!userSatisfactionFeedbacks.isEmpty()) {
            userSatisfaction = "Average: " + SystemLoadFragment.getAverage((CounterSet<Integer>)ratings) + ", Median: " + SystemLoadFragment.getMedian((CounterSet<Integer>)ratings) + " (" + userSatisfactionFeedbacks.size() + " feedback records)";
        }
        return userSatisfaction;
    }

    private static int getMedian(CounterSet<Integer> ratings) {
        List<Integer> expandingRatings = SystemLoadFragment.getExpandedRatingList(ratings);
        return expandingRatings.get(expandingRatings.size() / 2);
    }

    private static @NonNull List<Integer> getExpandedRatingList(CounterSet<Integer> ratings) {
        ArrayList<Integer> expandingRatings = new ArrayList<Integer>();
        for (Integer rating : ratings.getKeys()) {
            expandingRatings.addAll(CollectionUtils.repeat(i -> rating, (int)ratings.getValue((Object)rating)));
        }
        expandingRatings.sort(Integer::compareTo);
        return expandingRatings;
    }

    private static int getAverage(CounterSet<Integer> ratings) {
        return SystemLoadFragment.getExpandedRatingList(ratings).stream().mapToInt(i -> i).sum() / ratings.getTotal();
    }

    private static double determineCpuCoreCount(CentralProcessor cpuInfo) {
        if (OsUtils.isUNIX()) {
            long quota = FileUtil.getLongFromFile((String)"/sys/fs/cgroup/cpu/cpu.cfs_quota_us");
            long period = FileUtil.getLongFromFile((String)"/sys/fs/cgroup/cpu/cpu.cfs_period_us");
            if (quota > 0L && period > 0L) {
                return (double)quota / (double)period;
            }
            String cpuMax = FileUtil.getStringFromFile((String)"/sys/fs/cgroup/cpu.max");
            Matcher matcher = TWO_LONG_VALUES.matcher(cpuMax);
            if (matcher.matches()) {
                return (double)Long.parseLong(matcher.group(1)) / (double)Long.parseLong(matcher.group(2));
            }
        }
        return cpuInfo.getLogicalProcessorCount();
    }

    private static double determineCpuLoad(CentralProcessor cpuInfo, double coreCount, long[] previousTicks) {
        if (previousTicks.length == 2) {
            long[] currentTicks = SystemLoadFragment.determineCpuLoadTicks(null);
            long nanoDiff = currentTicks[0] - previousTicks[0];
            long ticksDiff = currentTicks[1] - previousTicks[1];
            if (nanoDiff <= 0L || ticksDiff <= 0L) {
                return 0.0;
            }
            return (double)ticksDiff / (double)nanoDiff / coreCount;
        }
        return cpuInfo.getSystemCpuLoadBetweenTicks(previousTicks);
    }

    public static long[] determineCpuLoadTicks(SystemInfo systemInfo) {
        if (OsUtils.isUNIX()) {
            long usage = FileUtil.getLongFromFile((String)"/sys/fs/cgroup/cpu/cpuacct.usage");
            if (usage > 0L) {
                return new long[]{System.nanoTime(), usage};
            }
            Map cpuMap = FileUtil.getKeyValueMapFromFile((String)"/sys/fs/cgroup/cpu.stat", (String)" ");
            String usageString = (String)cpuMap.get("usage_usec");
            if (usageString != null) {
                return new long[]{System.nanoTime(), Long.parseLong(usageString) * 1000L};
            }
        }
        return systemInfo.getHardware().getProcessor().getSystemCpuLoadTicks();
    }

    private static long getRamInstalledMB(SystemInfo systemInfo) {
        long installedMemory = 0L;
        try {
            installedMemory = SystemLoadFragment.checkForCgroup(systemInfo.getHardware().getMemory().getTotal(), "/sys/fs/cgroup/memory/memory.limit_in_bytes", "/sys/fs/cgroup/memory.max");
        }
        catch (Throwable t) {
            LOGGER.error("Error obtaining memory information with OSHI library: {}", (Object)t, (Object)t);
        }
        return SystemLoadFragment.byteToMegabyte(installedMemory);
    }

    private static long getRamUsedMB(SystemInfo systemInfo) {
        long usedMemory = 0L;
        try {
            usedMemory = SystemLoadFragment.checkForCgroup(systemInfo.getHardware().getMemory().getTotal() - systemInfo.getHardware().getMemory().getAvailable(), "/sys/fs/cgroup/memory/memory.usage_in_bytes", "/sys/fs/cgroup/memory.current");
        }
        catch (Throwable t) {
            LOGGER.error("Error obtaining memory information with OSHI library: {}", (Object)t, (Object)t);
        }
        return SystemLoadFragment.byteToMegabyte(usedMemory);
    }

    private static long checkForCgroup(long systemValue, String ... cgroupFilesToCheck) {
        if (!OsUtils.isUNIX()) {
            return systemValue;
        }
        for (String cgroupFile : cgroupFilesToCheck) {
            long cgroupLimit = FileUtil.getLongFromFile((String)cgroupFile);
            if (cgroupLimit <= 0L || cgroupLimit > systemValue) continue;
            return cgroupLimit;
        }
        return systemValue;
    }

    public static long getRamAvailableMB(SystemInfo systemInfo) {
        return Math.max(0L, SystemLoadFragment.getRamInstalledMB(systemInfo) - SystemLoadFragment.getRamUsedMB(systemInfo));
    }

    private static long determineFreeSpaceMB(File file) throws IOException {
        while (file != null) {
            if (Files.exists(file.toPath(), new LinkOption[0])) {
                return SystemLoadFragment.byteToMegabyte(Files.getFileStore(file.toPath()).getUsableSpace());
            }
            file = file.getParentFile();
        }
        return 0L;
    }

    private static long byteToMegabyte(long bytes) {
        return bytes / 0x100000L;
    }

    @Override
    public String getFragmentCaption() {
        return "System Load and Free Space";
    }

    @Override
    public int getFragmentOrder() {
        return 3;
    }

    @Override
    public List<SystemInfoEntry> convertToKeyValuePairs() {
        ArrayList<SystemInfoEntry> result = new ArrayList<SystemInfoEntry>();
        result.add(SystemInfoEntry.ofString("System time", this.systemTime));
        result.add(SystemInfoEntry.ofString("CPU", this.cpuDescription));
        result.add(SystemInfoEntry.ofString("CPU logical cores", String.valueOf(this.cpuCoreCount)));
        result.add(SystemInfoEntry.ofPercentage("CPU load", 100.0 * this.cpuLoad));
        result.add(SystemInfoEntry.ofString("RAM installed", SystemLoadFragment.convertToGB(this.ramInstalledMB, ByteUnit.MEBIBYTES)));
        result.add(SystemInfoEntry.ofString("RAM used", SystemLoadFragment.convertToGB(this.ramUsedMB, ByteUnit.MEBIBYTES)));
        double ramUsage = 100.0 * (double)this.ramUsedMB / Math.max(1.0, (double)this.ramInstalledMB);
        result.add(SystemInfoEntry.ofPercentage("RAM usage", ramUsage));
        result.addAll(SystemLoadFragment.listDirectoryInfo("Working directory", this.workingDirectory));
        if (this.storageDirectory != null) {
            result.addAll(SystemLoadFragment.listDirectoryInfo("Storage directory", this.storageDirectory));
            result.add(SystemInfoEntry.ofString("Storage directory total size", SystemLoadFragment.convertToGB(this.storageDirectorySize, ByteUnit.BYTES)));
        }
        result.addAll(SystemLoadFragment.listDirectoryInfo("Repository directory", this.repositoriesDirectory));
        result.addAll(SystemLoadFragment.listDirectoryInfo("Temporary directory", this.tempDirectory));
        result.add(SystemInfoEntry.ofString("User satisfaction", this.userSatisfaction));
        return result;
    }

    private static List<SystemInfoEntry> listDirectoryInfo(String prefix, PathWithSpaceInfo pathInfo) {
        return List.of(SystemInfoEntry.ofString(prefix, pathInfo.path), SystemInfoEntry.ofString(prefix + " free space", SystemLoadFragment.convertToGB(pathInfo.freeMB, ByteUnit.MEBIBYTES)));
    }

    public static String convertToGB(double value, ByteUnit unit) {
        double gibibytes = unit.toGibiBytes(value);
        return SystemInfoEntry.format(gibibytes) + " GB";
    }

    public static String convertToMB(double value, ByteUnit unit) {
        double mebibytes = unit.toMebiBytes(value);
        return SystemInfoEntry.format(mebibytes) + " MB";
    }

    public long getStorageDirectorySize() {
        return this.storageDirectorySize;
    }

    public double getCpuCoreCount() {
        return this.cpuCoreCount;
    }

    public double getCpuLoad() {
        return this.cpuLoad;
    }

    public long getRamInstalledMB() {
        return this.ramInstalledMB;
    }

    public long getRamUsedMB() {
        return this.ramUsedMB;
    }

    public PathWithSpaceInfo getWorkingDirectory() {
        return this.workingDirectory;
    }

    public @Nullable PathWithSpaceInfo getStorageDirectory() {
        return this.storageDirectory;
    }

    public PathWithSpaceInfo getRepositoriesDirectory() {
        return this.repositoriesDirectory;
    }

    public PathWithSpaceInfo getTempDirectory() {
        return this.tempDirectory;
    }

    @IndexValueClass
    public record PathWithSpaceInfo(String path, long freeMB) implements Serializable
    {
        private PathWithSpaceInfo(File path) throws IOException {
            this(path.getAbsolutePath(), SystemLoadFragment.determineFreeSpaceMB(path.getAbsoluteFile()));
        }
    }
}

