A URL that is accepted by an endpoint and directly used without checking its parameters, is possibly vulnerable to a Server-Side-Request-Forgery-Attack (SSRF).

# What Does This Check Look For?
A finding is created, if a REST controller method accepts a `string` or `Uri` object (these types are used for URLs) and first uses them to send a request using a `HttpClient` object.
You can avoid these findings by checking the properties of the URL before requesting data with it.
The following case would be flagged:

# Why is This a Problem?
Endpoints using unchecked URLs present a potential vulnerability to an SSRF attack, which can be exploited by malicious users.
By purposefully inserting a forged URL, an attacker can trick the program into making unintended calls.
If the URL is not meticulously validated, it opens up the possibility for the attacker to gain unauthorized access to sensitive data.
In a typical SSRF attack, the attacker may manipulate the server to establish connections with internal-only services within the organization's infrastructure or may coerce the server into connecting with arbitrary external systems.

# How Can I Resolve This?
1. Don't accept URLs from users.
2. Accept a destination key and use it to look up the target destination associated with the key.
3. Only allow certain domains or protocols by using an allow list.
4. Check the properties of the Uri object before its use.

## Example (Before)
```cs
[HttpGet]
public async Task<IActionResult> NonCompliant(Uri location)
{
	using HttpClient client = new HttpClient();
	var response = await client.GetAsync(location);
	return Ok();
}
```

## Example (After)
By adding an allow list of domains and protocols and comparing them with the received Uri object, the code could look like this:
```cs
private readonly string[] allowedSchemes = { "https" };
private readonly string[] allowedDomains = { "trusted1.example.com", "trusted2.example.com" };

[HttpGet]
public async Task<IActionResult> Compliant(Uri location)
{
	if (!allowedDomains.Contains(location.Host) || !allowedSchemes.Contains(location.Scheme))
	{
		return BadRequest("Invalid or disallowed URL.");
	}

	using HttpClient client = new HttpClient();
	var response = await client.GetAsync(location);
	return Ok();
}
```

# Where Can I Learn More?
- [OWASP: Server Side Request Forgery](https://owasp.org/www-community/attacks/Server_Side_Request_Forgery)
- [OWASP Cheat Sheet Series: Server-Side Request Forgery Prevention Cheat Sheet](https://cheatsheetseries.owasp.org/cheatsheets/Server_Side_Request_Forgery_Prevention_Cheat_Sheet.html)
