Disallow type aliases.


In TypeScript, type aliases serve three purposes:

- Aliasing other types so that we can refer to them using a simpler name.

```ts
// this...
type Person = {
    firstName: string,
    lastName: string,
    age: number
};

function addPerson(person : Person) { ... }

// is easier to read than this...
function addPerson(person : { firstName: string, lastName: string, age: number}) { ... }
```

- Act sort of like an interface, providing a set of methods and properties that must exist
  in the objects implementing the type.

```ts
type Person = {
    firstName: string,
    lastName: string,
    age: number,
    walk: () => void,
    talk: () => void
};

// you know person will have 3 properties and 2 methods,
// because the structure has already been defined.
var person : Person = { ... }

// so we can be sure that this will work
person.walk();
```

- Act like mapping tools between types to allow quick modifications.

```ts
type Immutable<T> = { readonly [P in keyof T]: T[P] };

type Person = {
  name: string;
  age: number;
};

type ImmutablePerson = Immutable<Person>;

var person: ImmutablePerson = { name: 'John', age: 30 };
person.name = 'Brad'; // error, readonly property
```

When aliasing, the type alias does not create a new type, it just creates a new name
to refer to the original type. So aliasing primitives and other simple types, tuples, unions
or intersections can some times be redundant.

```ts
// this doesn't make much sense
type myString = string;
```

On the other hand, using a type alias as an interface can limit your ability to:

- Reuse your code: interfaces can be extended or implemented by other types. Type aliases cannot.
- Debug your code: interfaces create a new name, so is easy to identify the base type of an object
  while debugging the application.

Finally, mapping types is an advanced technique and leaving it open can quickly become a pain point
in your application.

## Examples

This rule disallows the use of type aliases in favor of interfaces
and simplified types (primitives, tuples, unions, intersections, etc).

## When Not To Use It

When you can't express some shape with an interface or you need to use a union, tuple type,
callback, etc. that would cause the code to be unreadable or impractical.

## Further Reading

- [Advanced Types](https://www.typescriptlang.org/docs/handbook/advanced-types.html)
