<h2>Why is this an issue?</h2>
<p>JavaScript provides built-in methods to sort arrays, making it convenient for developers to manipulate data. There are two primary ways to sort an
array:</p>
<ul>
  <li> <code>Array.prototype.sort()</code>: This method sorts the elements of an array in place and returns the sorted array. </li>
  <li> <code>Array.prototype.toSorted()</code>: This method is designed to return a new sorted array, leaving the original array unchanged. </li>
</ul>
<p>The default sort order is lexicographic (dictionary) order, based on the string representation of the elements. This means that when sorting an
array of strings, numbers, or other elements, they are converted to strings and sorted according to their Unicode code points (UTF-16). For most
cases, this default behavior is suitable when sorting an array of strings.</p>
<p>However, it’s essential to be aware of potential pitfalls when sorting arrays of non-string elements, particularly numbers. The lexicographic order
may not always produce the expected results for numbers:</p>

```js
const numbers = [10, 2, 30, 1, 5];
numbers.sort(); // Noncompliant: lexicographic sort
console.log(numbers); // Output: [1, 10, 2, 30, 5]
```
<p>To sort numbers correctly, you must provide a custom comparison function that returns the correct ordering:</p>

```js
const numbers = [10, 2, 30, 1, 5];
numbers.sort((a, b) => a - b);
console.log(numbers); // Output: [1, 2, 5, 10, 30]
```
<p>Even to sort strings, the default sort order may give unexpected results. Not only does it not support localization, it also doesn’t fully support
Unicode, as it only considers UTF-16 code units. For example, in the code below, <code>"eΔ"</code> is surprisingly before and after <code>"éΔ"</code>.
To guarantee that the sorting is reliable and remains as such in the long run, it is necessary to provide a compare function that is both locale and
Unicode aware - typically <code>String.localeCompare</code>.</p>

```js
const code1 = '\u00e9\u0394'; // "éΔ"
const code2 = '\u0065\u0301\u0394'; // "éΔ" using Unicode combining marks
const code3 = '\u0065\u0394'; // "eΔ"
console.log([code1, code2, code3].sort()); // Noncompliant: ["éΔ", "eΔ", "éΔ"], "eΔ" position is inconsistent
console.log([code1, code2, code3].sort((a, b) => a.localeCompare(b))); // ["eΔ", "éΔ", "éΔ"]
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/sort"><code>Array.prototype.sort()</code></a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/toSorted"><code>Array.prototype.toSorted()</code></a>
  </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/localeCompare"><code>String.prototype.localeCompare()</code></a> </li>
</ul>