<p>The <code>&lt;a&gt;</code> tag in HTML is designed to create hyperlinks, which can link to different sections of the same page, different pages, or
even different websites. However, sometimes developers misuse <code>&lt;a&gt;</code> tags as buttons, which can lead to accessibility issues and
unexpected behavior.</p>
<p>This rule checks that <code>&lt;a&gt;</code> tags are used correctly as hyperlinks and not misused as buttons. It verifies that each
<code>&lt;a&gt;</code> tag has a <code>href</code> attribute, which is necessary for it to function as a hyperlink. If an <code>&lt;a&gt;</code> tag
is used without a <code>href</code> attribute, it behaves like a button, which is not its intended use.</p>
<p>Using the correct HTML elements for their intended purpose is crucial for accessibility and usability. It ensures that the website behaves as
expected and can be used by all users, including those using assistive technologies. Misusing HTML elements can lead to a poor user experience and
potential accessibility violations.</p>
<p>Compliance with this rule will ensure that your HTML code is semantically correct, accessible, and behaves as expected.</p>
<h2>Why is this an issue?</h2>
<p>Misusing <code>&lt;a&gt;</code> tags as buttons can lead to several issues:</p>
<ul>
  <li> Accessibility: Screen readers and other assistive technologies rely on the correct use of HTML elements to interpret and interact with the
  content. When <code>&lt;a&gt;</code> tags are used as buttons, it can confuse these technologies and make the website less accessible to users with
  disabilities. </li>
  <li> Usability: The behavior of <code>&lt;a&gt;</code> tags and buttons is different. For example, buttons can be triggered using the space bar,
  while <code>&lt;a&gt;</code> tags cannot. Misusing these elements can lead to unexpected behavior and a poor user experience. </li>
  <li> Semantic correctness: Each HTML element has a specific purpose and meaning. Using elements for purposes other than their intended use can make
  the code harder to understand and maintain. </li>
  <li> SEO implications: Search engines use the structure and semantics of HTML to understand and rank web pages. Misusing HTML elements can
  negatively impact a website’s SEO. </li>
</ul>
<h2>How to fix it</h2>
<p>To fix this issue, you should use the appropriate HTML elements for their intended purposes. If you need to create a hyperlink, use the
<code>&lt;a&gt;</code> tag with a <code>href</code> attribute. If you need to create a button, use the <code>&lt;button&gt;</code> tag.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```js
const MyComponent = () => {
    return <>
        <a href="javascript:void(0)" onClick={foo}>Perform action</a>
        <a href="#" onClick={foo}>Perform action</a>
        <a onClick={foo}>Perform action</a>
    </>;
};
```
<h4>Compliant solution</h4>

```js
const MyComponent = () => {
    return <>
      <button onClick={foo}>Perform action</button>
      <a href="#section" onClick={foo}>Perform action</a>
    </>;
};
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> WebAIM - <a href="https://webaim.org/techniques/hypertext/">Introduction to Links and Hypertext</a> </li>
  <li> <a href="https://marcysutton.com/links-vs-buttons-in-modern-web-applications/">Links vs. Buttons in Modern Web Applications</a> </li>
  <li> <a
  href="https://github.com/jsx-eslint/eslint-plugin-jsx-a11y/blob/main/docs/rules/anchor-is-valid.md#jsx-a11yanchor-is-valid:~:text=Using%20ARIA%20%2D%20Notes%20on%20ARIA%20use%20in%20HTML">Using ARIA - Notes on ARIA use in HTML</a> </li>
</ul>