<h2>Why is this an issue?</h2>
<p>When accessing <code>Map</code> elements by key, you should keep in mind that value might not be present. If value is not present,
<code>null</code> will be returned. To make it possible, the type of returned value is nullable. In Kotlin, it’s not usually convenient to operate
with nullable types, so developers usually try to avoid them or convert them to non-nullable types. One of the possible solutions is to use
<code>!!</code> (non-null assertion operator). If during the runtime the actual value applied to non-null asserrion operator was <code>null</code>,
then <code>NullPointerException</code> will be thrown. While in some cases it could still be legitimate to use <code>!!</code>, accesing
<code>Map</code> values is not one of them. Usage of a <code>!!</code> when accesing <code>Map</code> values is a bad practice and can lead to
<code>NullPointerException</code> in Kotlin and potential crashes, if Java interop was involved.</p>
<h2>How to fix it</h2>
<p>You should avoid using the non-null assertion operator when accessing <code>Map</code> elements.</p>
<p>Instead, you can:</p>
<ul>
  <li> return an instance of a nullable type. </li>
  <li> use <code>getValue</code>. This will throw <code>NoSuchElementException</code> when an element is not present. </li>
  <li> use either <code>getOrElse</code> or <code>getOrDefault</code> functions and explicitly specify the behavior in case of a null value. </li>
  <li> use an elvis operator <code>?:</code> to specify the behavior in case of a null value. </li>
</ul>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>This example will throw a <code>NullPointerException</code> at the <code>!!</code> operator because <code>map</code> doesn’t have a key
<code>123</code> and the result of <code>get(123)</code> will be null.</p>

```kotlin
val l = mapOf(1 to "one", 2 to "two", 3 to "five")
l.get(123)!! // Noncompliant
l[123]!! // Noncompliant
```
<h4>Compliant solution</h4>
<p>By removing the non-null assertion operator the result of the <code>get(123)</code> call will return an instance of a nullable type, and the user
can handle the potential <code>null</code> value properly.</p>

```kotlin
val l = mapOf(1 to "one", 2 to "two", 3 to "five")
l.get(123) // Compliant, returns nullable
l[123] // Compliant, returns nullable
l.getValue(123) // Compliant, throws NoSuchElementException
l.getOrElse(123 ) { "empty" } // Compliant, has default
l.getOrDefault(123, "empty") // Compliant, has default
l[123] ?: "empty" // Compliant, has default
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://kotlinlang.org/docs/null-safety.html">Kotlin Documentation - Null safety</a> </li>
  <li> <a href="https://kotlinlang.org/docs/map-operations.html">Kotlin Documentation - Map-specific operations</a> </li>
  <li> <a href="https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/get-or-else.html">Kotlin Documentation - getOrElse</a> </li>
  <li> <a href="https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/get-or-default.html">Kotlin Documentation - getOrDefault</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://medium.com/@igorwojda/kotlin-combating-non-null-assertions-5282d7b97205">Kotlin — combating non-null assertions (!!)</a> </li>
</ul>