<h2>Why is this an issue?</h2>
<p>Dead stores refer to assignments made to local variables that are subsequently never used or immediately overwritten. Such assignments are
unnecessary and don’t contribute to the functionality or clarity of the code. They may even negatively impact performance. Removing them enhances code
cleanliness and readability. Even if the unnecessary operations do not do any harm in terms of the program’s correctness, they are - at best - a waste
of computing resources.</p>
<h3>Exceptions</h3>
<p>The rule ignores</p>
<ul>
  <li> Initializations to <code>-1</code>, <code>0</code>, <code>1</code>, <code>undefined</code>, <code>[]</code>, <code>{}</code>,
  <code>true</code>, <code>false</code> and <code>""</code>. </li>
  <li> Variables that start with an underscore (e.g. <code>_unused</code>) are ignored. </li>
  <li> Assignment of <code>null</code> is ignored because it is sometimes used to help garbage collection </li>
  <li> Increment and decrement expressions are ignored because they are often used idiomatically instead of <code>x+1</code> </li>
  <li> This rule also ignores variables declared with object destructuring using rest syntax (used to exclude some properties from object) </li>
</ul>

```ts
let {a, b, ...rest} = obj;  // 'a' and 'b' are compliant
doSomething(rest);

let [x1, x2, x3] = arr;     // 'x1' is noncompliant, as omitting syntax can be used: "let [, x2, x3] = arr;"
doSomething(x2, x3);
```
<h2>How to fix it</h2>
<p>Remove the unnecesarry assignment, then test the code to make sure that the right-hand side of a given assignment had no side effects (e.g. a
method that writes certain data to a file and returns the number of written bytes).</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```ts
function foo(y) {
  let x = 100; // Noncompliant: dead store
  x = 150;     // Noncompliant: dead store
  x = 200;
  return x + y;
}
```
<h4>Compliant solution</h4>

```ts
function foo(y) {
  let x = 200; // Compliant: no unnecessary assignment
  return x + y;
}
```
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/563">CWE-563 - Assignment to Variable without Use ('Unused Variable')</a> </li>
</ul>
<h3>Related rules</h3>
<ul>
  <li> <a href='checks/checks/?languages=JAVASCRIPT&tools=SONAR_LINT&rule=javascript:S1763'>javascript:S1763</a> - All code should be reachable </li>
  <li> <a href='checks/checks/?languages=JAVASCRIPT&tools=SONAR_LINT&rule=javascript:S2589'>javascript:S2589</a> - Boolean expressions should not be gratuitous </li>
  <li> <a href='checks/checks/?languages=JAVASCRIPT&tools=SONAR_LINT&rule=javascript:S3516'>javascript:S3516</a> - Function returns should not be invariant </li>
  <li> <a href='checks/checks/?languages=JAVASCRIPT&tools=SONAR_LINT&rule=javascript:S3626'>javascript:S3626</a> - Jump statements should not be redundant </li>
</ul>