<h2>Why is this an issue?</h2>
<p>React Refs provide a way to access DOM nodes or React elements created in the render method.</p>
<p>Older React versions allowed the ref attribute to be a string, like <code>"textInput"</code>, later accessible as <code>this.refs.textInput</code>.
This is considered legacy code due to multiple reasons:</p>
<ul>
  <li> String <code>refs</code> make React slower as they force React to keep track of what component is currently executing. </li>
  <li> String <code>refs</code> are not composable: if a library puts a ref on the passed child, the user can’t put another ref on it. </li>
  <li> The owner of a string <code>ref</code> is determined by the currently executing component. </li>
</ul>

```ts
const Hello = createReactClass({
  componentDidMount() {
    const component = this.refs.hello; // Noncompliant
    // ...
  },
  render() {
    return <div ref="hello">Hello, world.</div>;
  }
});
```
<p>Instead, reference callbacks should be used. These do not have the limitations mentioned above. When the DOM node is added to the screen, React
will call the <code>ref</code> callback with the DOM node as the argument. When that DOM node is removed, React will call your <code>ref</code>
callback with <code>null</code>. One should return <code>undefined</code> from the <code>ref</code> callback.</p>

```ts
const Hello = createReactClass({
  componentDidMount() {
    const component = this.hello;
    // ...
  },
  render() {
    return <div ref={(c) => { this.hello = c; }}>Hello, world.</div>;
  }
});
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a href="https://react.dev/reference/react-dom/components/common#ref-callback"><code>ref</code> callback function</a>
  </li>
  <li> React Documentation - <a href="https://legacy.reactjs.org/docs/refs-and-the-dom.html">Refs and the DOM</a> </li>
  <li> React Documentation - <a href="https://react.dev/learn/manipulating-the-dom-with-refs">Manipulating the DOM with Refs</a> </li>
  <li> React Documentation - <a href="https://react.dev/reference/react/useRef"><code>useRef</code></a> </li>
  <li> React Documentation - <a href="https://react.dev/reference/react/createRef#createref"><code>createRef</code></a> </li>
</ul>