"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RULE_NAME = void 0;
const utils_1 = require("@angular-eslint/utils");
const create_eslint_rule_1 = require("../utils/create-eslint-rule");
const is_child_node_of_1 = require("../utils/is-child-node-of");
exports.RULE_NAME = 'label-has-associated-control';
const DEFAULT_CONTROL_COMPONENTS = [
    'input',
    'meter',
    'output',
    'progress',
    'select',
    'textarea',
];
const DEFAULT_LABEL_COMPONENTS = [
    { inputs: ['for', 'htmlFor'], selector: 'label' },
];
const DEFAULT_OPTIONS = {
    controlComponents: DEFAULT_CONTROL_COMPONENTS,
    labelComponents: DEFAULT_LABEL_COMPONENTS,
};
exports.default = (0, create_eslint_rule_1.createESLintRule)({
    name: exports.RULE_NAME,
    meta: {
        type: 'suggestion',
        docs: {
            description: '[Accessibility] Ensures that a label element/component is associated with a form element',
        },
        schema: [
            {
                additionalProperties: false,
                properties: {
                    controlComponents: {
                        items: { type: 'string' },
                        type: 'array',
                        uniqueItems: true,
                    },
                    labelComponents: {
                        items: {
                            additionalProperties: false,
                            properties: {
                                inputs: {
                                    items: { type: 'string' },
                                    type: 'array',
                                    uniqueItems: true,
                                },
                                selector: { type: 'string' },
                            },
                            required: ['selector'],
                            type: 'object',
                        },
                        type: 'array',
                        uniqueItems: true,
                    },
                },
                type: 'object',
            },
        ],
        messages: {
            labelHasAssociatedControl: 'A label component must be associated with a form element',
        },
    },
    defaultOptions: [DEFAULT_OPTIONS],
    create(context, [{ controlComponents, labelComponents }]) {
        const parserServices = (0, utils_1.getTemplateParserServices)(context);
        const allControlComponents = new Set([
            ...DEFAULT_CONTROL_COMPONENTS,
            ...(controlComponents !== null && controlComponents !== void 0 ? controlComponents : []),
        ]);
        const allLabelComponents = [
            ...DEFAULT_LABEL_COMPONENTS,
            ...(labelComponents !== null && labelComponents !== void 0 ? labelComponents : []),
        ];
        const labelSelectors = allLabelComponents.map(({ selector }) => selector);
        const labelComponentsPattern = toPattern(labelSelectors);
        return {
            [`Element$1[name=${labelComponentsPattern}]`](node) {
                var _a;
                const element = allLabelComponents.find(({ selector }) => selector === node.name);
                if (!element)
                    return;
                const attributesInputs = new Set([...node.attributes, ...node.inputs].map(({ name }) => name));
                const hasInput = (_a = element.inputs) === null || _a === void 0 ? void 0 : _a.some((input) => attributesInputs.has(input));
                if (hasInput || hasControlComponentIn(allControlComponents, node)) {
                    return;
                }
                const loc = parserServices.convertNodeSourceSpanToLoc(node.sourceSpan);
                context.report({
                    loc,
                    messageId: 'labelHasAssociatedControl',
                });
            },
        };
    },
});
function hasControlComponentIn(controlComponents, element) {
    return Boolean([...controlComponents].some((controlComponent) => (0, is_child_node_of_1.isChildNodeOf)(element, controlComponent)));
}
function toPattern(value) {
    return RegExp(`^(${value.join('|')})$`);
}
