/*
 * Decompiled with CFR 0.152.
 */
package com.teamscale.core.rest.client.retry;

import com.teamscale.core.rest.client.retry.HttpRequestRetryPolicy;
import java.io.IOException;
import java.time.Duration;
import okhttp3.Interceptor;
import okhttp3.Request;
import okhttp3.Response;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.conqat.lib.commons.date.DurationUtils;
import org.jetbrains.annotations.VisibleForTesting;
import org.jspecify.annotations.NonNull;

class HttpRequestRetryInterceptor
implements Interceptor {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final int EXPONENTIAL_BACKOFF_FACTOR = 2;
    private static final Duration MAX_SLEEP_TIME = Duration.ofHours(6L);
    private final HttpRequestRetryPolicy retryPolicy;
    private final ISleeper sleeper;

    HttpRequestRetryInterceptor(HttpRequestRetryPolicy retryPolicy) {
        this.retryPolicy = retryPolicy;
        this.sleeper = this::defaultSleep;
    }

    HttpRequestRetryInterceptor(HttpRequestRetryPolicy retryPolicy, ISleeper sleeper) {
        this.retryPolicy = retryPolicy;
        this.sleeper = sleeper;
    }

    public @NonNull Response intercept(Interceptor.Chain chain) throws IOException {
        Request request = chain.request();
        try {
            return chain.proceed(request);
        }
        catch (IOException e) {
            if (this.retryPolicy.getNumberOfRetries() == 0 || !this.retryPolicy.isExceptionToRetry(e)) {
                throw e;
            }
            LOGGER.warn("API request {} suffered a connection failure. Will retry {} times.", (Object)request.url(), (Object)this.retryPolicy.getNumberOfRetries(), (Object)e);
            return this.retryRequestWithExponentialBackoff(chain, request);
        }
    }

    private Response retryRequestWithExponentialBackoff(Interceptor.Chain chain, Request request) throws IOException {
        IOException lastCaughtException = null;
        for (int i = 0; i < this.retryPolicy.getNumberOfRetries(); ++i) {
            Duration waitDuration = this.retryPolicy.getInitialDelay().multipliedBy((long)Math.pow(2.0, i));
            if (waitDuration.compareTo(MAX_SLEEP_TIME) > 0) {
                LOGGER.warn("Refusing to sleep for " + DurationUtils.formatDurationHumanReadable((Duration)waitDuration) + " which is longer than the maximum sleep time of " + DurationUtils.formatDurationHumanReadable((Duration)MAX_SLEEP_TIME));
                waitDuration = MAX_SLEEP_TIME;
            }
            LOGGER.warn("Waiting for " + DurationUtils.formatDurationHumanReadable((Duration)waitDuration) + " before retrying request");
            this.sleeper.sleep(waitDuration);
            try {
                return chain.proceed(request);
            }
            catch (IOException e) {
                if (!this.retryPolicy.isExceptionToRetry(e)) {
                    throw e;
                }
                lastCaughtException = e;
                int retryNumber = i + 1;
                LOGGER.warn("API request {} suffered a connection failure (again) during retry #{}. Will retry {} more times.", (Object)request.url(), (Object)retryNumber, (Object)(this.retryPolicy.getNumberOfRetries() - retryNumber), (Object)e);
                continue;
            }
        }
        throw new IOException("Connection problems even after retries: Failed to complete request " + String.valueOf(request.url()) + " even after " + this.retryPolicy.getNumberOfRetries() + " retries", lastCaughtException);
    }

    private void defaultSleep(Duration duration) {
        try {
            Thread.sleep(duration.toMillis());
        }
        catch (InterruptedException e) {
            LOGGER.error("Retry thread was unexpectedly interrupted. Retrying early", (Throwable)e);
        }
    }

    @FunctionalInterface
    @VisibleForTesting
    static interface ISleeper {
        public void sleep(Duration var1);
    }
}

