<p>Because it is easy to extract strings from an application source code or binary, credentials should not be hard-coded. This is particularly true
for applications that are distributed or that are open-source.</p>
<p>In the past, it has led to the following vulnerabilities:</p>
<ul>
  <li> <a href="http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2019-13466">CVE-2019-13466</a> </li>
  <li> <a href="http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-15389">CVE-2018-15389</a> </li>
</ul>
<p>Credentials should be stored outside of the code in a configuration file, a database, or a management service for secrets.</p>
<p>This rule flags instances of hard-coded credentials used in database and LDAP connections. It looks for hard-coded credentials in connection
strings, and for variable names that match any of the patterns from the provided list.</p>
<p>It’s recommended to customize the configuration of this rule with additional credential words such as "oauthToken", "secret", …​</p>
<h2>Ask Yourself Whether</h2>
<ul>
  <li> Credentials allow access to a sensitive component like a database, a file storage, an API or a service. </li>
  <li> Credentials are used in production environments. </li>
  <li> Application re-distribution is required before updating the credentials. </li>
</ul>
<p>There is a risk if you answered yes to any of those questions.</p>
<h2>Recommended Secure Coding Practices</h2>
<ul>
  <li> Store the credentials in a configuration file that is not pushed to the code repository. </li>
  <li> Store the credentials in a database. </li>
  <li> Use your cloud provider’s service for managing secrets. </li>
  <li> If a password has been disclosed through the source code: change it. </li>
</ul>
<h2>Sensitive Code Example</h2>
<pre>
string username = "admin";
string password = "Admin123"; // Sensitive
string usernamePassword  = "user=admin&amp;password=Admin123"; // Sensitive
string url = "scheme://user:Admin123@domain.com"; // Sensitive
</pre>
<h2>Compliant Solution</h2>
<pre>
string username = "admin";
string password = GetEncryptedPassword();
string usernamePassword = string.Format("user={0}&amp;password={1}", GetEncryptedUsername(), GetEncryptedPassword());
string url = $"scheme://{username}:{password}@domain.com";

string url2 = "http://guest:guest@domain.com"; // Compliant
const string Password_Property = "custom.password"; // Compliant
</pre>
<h2>Exceptions</h2>
<ul>
  <li> Issue is not raised when URI username and password are the same. </li>
  <li> Issue is not raised when searched pattern is found in variable name and value. </li>
</ul>
<h2>See</h2>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/">Top 10 2021 Category A7 - Identification and
  Authentication Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A2_2017-Broken_Authentication">Top 10 2017 Category A2 - Broken Authentication</a>
  </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/798">CWE-798 - Use of Hard-coded Credentials</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/259">CWE-259 - Use of Hard-coded Password</a> </li>
  <li> Derived from FindSecBugs rule <a href="https://h3xstream.github.io/find-sec-bugs/bugs.htm#HARD_CODE_PASSWORD">Hard Coded Password</a> </li>
</ul>

