<p>Using pseudorandom number generators (PRNGs) is security-sensitive. For example, it has led in the past to the following vulnerabilities:</p>
<ul>
  <li> <a href="http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2013-6386">CVE-2013-6386</a> </li>
  <li> <a href="http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2006-3419">CVE-2006-3419</a> </li>
  <li> <a href="http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2008-4102">CVE-2008-4102</a> </li>
</ul>
<p>When software generates predictable values in a context requiring unpredictability, it may be possible for an attacker to guess the next value that
will be generated, and use this guess to impersonate another user or access sensitive information.</p>
<p>As the <code>System.Random</code> class relies on a pseudorandom number generator, it should not be used for security-critical applications or for
protecting sensitive data. In such context, the <code>System.Cryptography.RandomNumberGenerator</code> class which relies on a cryptographically
strong random number generator (RNG) should be used in place.</p>
<h2>Ask Yourself Whether</h2>
<ul>
  <li> the code using the generated value requires it to be unpredictable. It is the case for all encryption mechanisms or when a secret value, such
  as a password, is hashed. </li>
  <li> the function you use generates a value which can be predicted (pseudo-random). </li>
  <li> the generated value is used multiple times. </li>
  <li> an attacker can access the generated value. </li>
</ul>
<p>There is a risk if you answered yes to any of those questions.</p>
<h2>Recommended Secure Coding Practices</h2>
<ul>
  <li> Only use random number generators which are <a
  href="https://cheatsheetseries.owasp.org/cheatsheets/Cryptographic_Storage_Cheat_Sheet.html#secure-random-number-generation">recommended by
  OWASP</a> or any other trusted organization. </li>
  <li> Use the generated random values only once. </li>
  <li> You should not expose the generated random value. If you have to store it, make sure that the database or file is secure. </li>
</ul>
<h2>Sensitive Code Example</h2>
<pre>
var random = new Random(); // Sensitive use of Random
byte[] data = new byte[16];
random.NextBytes(data);
return BitConverter.ToString(data); // Check if this value is used for hashing or encryption
</pre>
<h2>Compliant Solution</h2>
<pre>
using System.Security.Cryptography;
...
var randomGenerator = RandomNumberGenerator.Create(); // Compliant for security-sensitive use cases
byte[] data = new byte[16];
randomGenerator.GetBytes(data);
return BitConverter.ToString(data);
</pre>
<h2>See</h2>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">Top 10 2021 Category A2 - Cryptographic Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> <a href="https://mas.owasp.org/checklists/MASVS-CRYPTO/">Mobile AppSec Verification Standard - Cryptography Requirements</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-mobile-top-10/2016-risks/m5-insufficient-cryptography">Mobile Top 10 2016 Category M5 -
  Insufficient Cryptography</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/338">CWE-338 - Use of Cryptographically Weak Pseudo-Random Number Generator (PRNG)</a>
  </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/330">CWE-330 - Use of Insufficiently Random Values</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/326">CWE-326 - Inadequate Encryption Strength</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/1241">CWE-1241 - Use of Predictable Algorithm in Random Number Generator</a> </li>
  <li> Derived from FindSecBugs rule <a href="https://h3xstream.github.io/find-sec-bugs/bugs.htm#PREDICTABLE_RANDOM">Predictable Pseudo Random Number
  Generator</a> </li>
</ul>

