<h2>Why is this an issue?</h2>
<p>In the interests of readability, code that can be simplified should be simplified. To that end, there are several ways <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.ienumerable-1">IEnumerable</a> language integrated queries (LINQ) can be
simplified. This not only improves readabilty but can also lead to improved performance.</p>
<h2>How to fix it</h2>
<p>Simplify the LINQ expressions:</p>
<ul>
  <li> Use <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.oftype">OfType</a> instead of <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.select">Select</a> with the <a
  href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast#as-operator">as operator</a> to type cast
  elements and then null-checking in a query expression to choose elements based on type. </li>
  <li> Use <code>OfType</code> instead of using <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.where">Where</a> and the
  <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast#is-operator">is operator</a>, followed
  by a cast in a <code>Select</code> </li>
  <li> Use an expression in <code>Any</code> instead of <code>Where(element ⇒ [expression]).Any()</code>. </li>
  <li> Use the <a href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.list-1.count">Count</a> or <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.array.length">Length</a> properties instead of the <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.count">Count() method</a> when it’s available (unless you use the
  predicate parameter of the method for filtering). </li>
  <li> Don’t call <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.toarray">ToArray()</a> or <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.tolist">ToList()</a> in the middle of a query chain. </li>
</ul>
<p>Using <a href="https://learn.microsoft.com/en-us/ef/">Entity Framework</a> may require enforcing client evaluations. Such queries should use <a
href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.asenumerable">AsEnumerable()</a> instead of <code>ToArray()</code> or
<code>ToList()</code> in the middle of a query chain.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
public void Foo(IEnumerable&lt;Vehicle&gt; seq, List&lt;int&gt; list)
{
    var result1 = seq.Select(x =&gt; x as Car).Any(x =&gt; x != null);               // Noncompliant; use OfType
    var result2 = seq.Select(x =&gt; x as Car).Any(x =&gt; x != null &amp;&amp; x.HasOwner); // Noncompliant; use OfType before calling Any
    var result3 = seq.Where(x =&gt; x is Car).Select(x =&gt; x as Car);              // Noncompliant; use OfType
    var result4 = seq.Where(x =&gt; x is Car).Select(x =&gt; (Car)x);                // Noncompliant; use OfType
    var result5 = seq.Where(x =&gt; x.HasOwner).Any();                            // Noncompliant; use Any([predicate])

    var num = list.Count();                                                    // Noncompliant; use the Count property
    var arr = seq.ToList().ToArray();                                          // Noncompliant; ToList is not needed
    var count = seq.ToList().Count(x =&gt; x.HasOwner);                           // Noncompliant; ToList is not needed
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
public void Foo(IEnumerable&lt;Vehicle&gt; seq, List&lt;int&gt; list)
{
    var result1 = seq.OfType&lt;Car&gt;().Any();
    var result2 = seq.OfType&lt;Car&gt;().Any(x =&gt; x.HasOwner);
    var result3 = seq.OfType&lt;Car&gt;();
    var result4 = seq.OfType&lt;Car&gt;();
    var result5 = seq.Any(x =&gt; x.HasOwner);

    var num = list.Count;
    var arr = seq.ToArray();
    var count = seq.Count(x =&gt; x.HasOwner);
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/linq">Language Integrated Queries in C#</a> </li>
</ul>

