<h2>Why is this an issue?</h2>
<p><code>ThreadLocal</code> variables are supposed to be garbage collected once the holding thread is no longer alive. Memory leaks can occur when
holding threads are re-used which is the case on application servers using pool of threads.</p>
<p>To avoid such problems, it is recommended to always clean up <code>ThreadLocal</code> variables using the <code>remove()</code> method to remove
the current thread’s value for the <code>ThreadLocal</code> variable.</p>
<p>In addition, calling <code>set(null)</code> to remove the value might keep the reference to <code>this</code> pointer in the map, which can cause
memory leak in some scenarios. Using <code>remove</code> is safer to avoid this issue.</p>
<h3>Noncompliant code example</h3>

```java
public class ThreadLocalUserSession implements UserSession {

  private static final ThreadLocal<UserSession> DELEGATE = new ThreadLocal<>();

  public UserSession get() {
    UserSession session = DELEGATE.get();
    if (session != null) {
      return session;
    }
    throw new UnauthorizedException("User is not authenticated");
  }

  public void set(UserSession session) {
    DELEGATE.set(session);
  }

   public void incorrectCleanup() {
     DELEGATE.set(null); // Noncompliant
   }

  // some other methods without a call to DELEGATE.remove()
}
```
<h3>Compliant solution</h3>

```java
public class ThreadLocalUserSession implements UserSession {

  private static final ThreadLocal<UserSession> DELEGATE = new ThreadLocal<>();

  public UserSession get() {
    UserSession session = DELEGATE.get();
    if (session != null) {
      return session;
    }
    throw new UnauthorizedException("User is not authenticated");
  }

  public void set(UserSession session) {
    DELEGATE.set(session);
  }

  public void unload() {
    DELEGATE.remove(); // Compliant
  }

  // ...
}
```
<h3>Exceptions</h3>
<p>Rule will not detect non-private <code>ThreadLocal</code> variables, because <code>remove()</code> can be called from another class.</p>
<h2>Resources</h2>
<ul>
  <li> <a href="https://www.baeldung.com/java-memory-leaks">Understanding Memory Leaks in Java</a> </li>
</ul>