<h2>Why is this an issue?</h2>
<p>Promises are objects that represent the eventual completion or failure of an asynchronous operation. They provide a way to handle asynchronous
operations in a more organized and manageable manner. To use <code>await</code>, you need to ensure that you are calling a function or an expression
that returns a promise.</p>
<p>The reason <code>await</code> should only be used on a promise is that it expects the operand to be a promise object. When you use
<code>await</code>, it waits for the promise to be resolved or rejected and then returns the resolved value or throws the rejection reason,
respectively. If the operand of <code>await</code> is not a promise, awaiting it is redundant and might not have been the developer’s intent.</p>
<p>If you try to use <code>await</code> on a non-promise value, such as a regular object or a primitive type, it will not pause the execution of the
function because there is no asynchronous behavior involved. Instead, <code>await</code> will convert the value to a resolved promise, and waits for
it.</p>

```js
const x = 42;
await x; // Noncompliant: x is a number, not a promise
```
<p>You should only use <code>await</code> on promises because it is designed to handle asynchronous operations and works in conjunction with the
Promise API to provide a clean and intuitive way to write asynchronous code in JavaScript.</p>

```js
const x = Promise.resolve(42);
await x;
```
<p>When calling a function that returns a promise as the last expression, you might forget to return it, especially if you refactored your code from a
single-expression arrow function.</p>

```js
function foo() {
  Promise.resolve(42);
}
async function bar() {
  await foo(); // Noncompliant
}
```
<p>Make sure that you return the promise.</p>

```js
function foo() {
  return Promise.resolve(42);
}
async function bar() {
  await foo(); // Compliant
}
```
<h3>Exceptions</h3>
<p>The rule does not raise issues if you are awaiting a function whose definition contains JSdoc with <code>@returns</code> or <code>@return</code>
tags. This is due to JSdoc often mistakenly declaring a returning type without mentioning that it is resolved by a promise. For example:</p>

```js
async function foo () {
  await bar(); // Compliant
}

/**
 * @return {number}
 */
async function bar () {
  return 42;
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/await">await</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise">Promise</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/await#conversion_to_promise">Conversion to
  promise</a> </li>
</ul>