<h2>Why is this an issue?</h2>
<p>Promises need to be resolved or awaited to return the expected value, otherwise, they return the promise object.</p>
<p><span><strong>Unresolved promises:</strong></span></p>
<p>Forgetting to await a promise is a frequent mistake. There are places where the use of a promise object is confusing or unclear because the
developer forgot to resolve it.</p>
<p>This rule forbids returning promises where another type is expected such as in:</p>
<ul>
  <li> conditionals </li>
  <li> void returns </li>
  <li> spread operators </li>
</ul>
<h3>What is the potential impact?</h3>
<p>Using a promise instead of its resolved value can have unexpected results leading to bugs.</p>
<ul>
  <li> In conditionals, it will always return a truthy value. </li>
  <li> In places where the expected type is void, returning a promise is often a mistake. </li>
  <li> Using the spread operator on a promise will raise an exception. </li>
</ul>
<p>The executor function of a promise can also be an async function. However, this usually denotes a mistake:</p>
<ul>
  <li> If an async executor function throws an error, the error won’t cause the created promise to reject and will be lost. Therefore, this could make
  it difficult to debug and handle runtime errors. </li>
  <li> If a promise executor function is using <code>await</code>, this means that it’s not necessary to use the <code>Promise</code> constructor, or
  the scope of the Promise constructor can be reduced. </li>
</ul>
<h3>Exceptions</h3>
<p>This rule can be ignored for promises that you know will always resolve like timers.</p>

```js
await new Promise(resolve => time.setTimeout(1000));
```
<h2>How to fix it</h2>
<p>If you mistakenly treated a promise as its resolved value, you can ensure it is properly resolved by using await or resolve on the promise. In some
cases, you may need to use an "immediately invoked function expression" (IIFE):</p>

```js
(async function foo() {
  const result = await bar();
  // work with result
})();
```
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```js
const promise = new Promise((resolve, reject) => {
  // ...
  resolve(false)
});
if (promise) {
  // ...
}
```
<h4>Compliant solution</h4>

```js
const promise = new Promise((resolve, reject) => {
  // ...
  resolve(false)
});
if (await promise) {
  // ...
}
```
<h4>Noncompliant code example</h4>

```js
const p = new Promise(async (resolve, reject) => {
  doSomething('Hey, there!', function(error, result) {
    if (error) {
      reject(error);
      return;
    }
    await saveResult(result)
    resolve(result);
  });
});

await p;
```
<h4>Compliant solution</h4>

```js
const p = new Promise((resolve, reject) => {
  doSomething('Hey, there!', function(error, result) {
    if (error) {
      reject(error);
      return;
    }
    resolve(result);
  });
});

const result = await p;
await saveResult(result);
```
<h4>Noncompliant code example</h4>

```js
apiCalls.forEach(async (apiCall) => {
  await apiCall.send();
});
```
<h4>Compliant solution</h4>

```js
for (const apiCall of apiCalls) {
  await apiCall.send();
}
```
<h3>How does this work?</h3>
<p>In JavaScript, a promise is a mechanism to perform tasks asynchronously. To this end, the language provides the <code>Promise</code> object which
represents the eventual completion or failure of an asynchronous operation and its resulting value. A promise can be created with the
<code>Promise</code> constructor accepting an executor function as an argument, which has <code>resolve</code> and <code>reject</code> parameters that
are invoked when the promise completes or fails.</p>
<p>The logic of the promise is executed when it is called, however, its result is obtained only when the promise is resolved or awaited.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise">Promise</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Using_promises">Using promises</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/async_function">Async function</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Glossary/IIFE">IIFE</a> </li>
</ul>