<h2>Why is this an issue?</h2>
<p>In JavaScript, props are typically passed as plain objects, which can lead to errors and confusion when working with components that have specific
prop requirements. However, it lacks of type safety and clarity when passing props to components in a codebase.</p>
<p>By defining types for component props, developers can enforce type safety and provide clear documentation for the expected props of a component.
This helps catch potential errors at compile-time. It also improves code maintainability by making it easier to understand how components should be
used and what props they accept.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```js
import PropTypes from 'prop-types';

function Hello({ firstname, lastname }) {
  return <div>Hello {firstname} {lastname}</div>; // Noncompliant: 'lastname' type is missing
}
Hello.propTypes = {
  firstname: PropTypes.string.isRequired
};

// Using legacy APIs

class Hello extends React.Component {
  render() {
    return <div>Hello {this.props.firstname} {this.props.lastname}</div>; // Noncompliant: 'lastname' type is missing
  }
}
Hello.propTypes = {
  firstname: PropTypes.string.isRequired,
};
```
<h4>Compliant solution</h4>

```js
import PropTypes from 'prop-types';

function Hello({ firstname, lastname }) {
  return <div>Hello {firstname} {lastname}</div>;
}
Hello.propTypes = {
  firstname: PropTypes.string.isRequired,
  lastname: PropTypes.string.isRequired,
};

// Using legacy APIs

class Hello extends React.Component {
  render() {
    return <div>Hello {this.props.firstname} {this.props.lastname}</div>;
  }
}
Hello.propTypes = {
  firstname: PropTypes.string.isRequired,
  lastname: PropTypes.string.isRequired,
};
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> React Documentation - <a href="https://react.dev/reference/react/Component#static-proptypes">static propTypes</a> </li>
  <li> Flow.js Documentation - <a href="https://flow.org/en/docs/react/">React</a> </li>
</ul>