<p>Local variables and function parameters should be named consistently to communicate intent and improve maintainability. Rename your local variable
or function parameter to follow your project’s naming convention to address this issue.</p>
<h2>Why is this an issue?</h2>
<p>A naming convention in software development is a set of guidelines for naming code elements like variables, functions, and classes.<br> Local
variables and function parameters hold the meaning of the written code. Their names should be meaningful and follow a consistent and easily
recognizable pattern.<br> Adhering to a consistent naming convention helps to make the code more readable and understandable, which makes it easier to
maintain and debug. It also ensures consistency in the code, especially when multiple developers are working on the same project.</p>
<p>This rule checks that local variable and function parameter names match a provided regular expression.</p>
<h3>What is the potential impact?</h3>
<p>Inconsistent naming of local variables and function parameters can lead to several issues in your code:</p>
<ul>
  <li> Reduced Readability: inconsistent local variable and function parameter names make the code harder to read and understand; consequently, it is
  more difficult to identify the purpose of each variable, spot errors, or comprehend the logic. </li>
  <li> Difficulty in Identifying Variables: local variables and function parameters that don’t adhere to a standard naming convention are challenging
  to identify; thus, the coding process slows down, especially when dealing with a large codebase. </li>
  <li> Increased Risk of Errors: inconsistent or unclear local variable and function parameter names lead to misunderstandings about what the variable
  represents. This ambiguity leads to incorrect assumptions and, consequently, bugs in the code. </li>
  <li> Collaboration Difficulties: in a team setting, inconsistent naming conventions lead to confusion and miscommunication among team members. </li>
  <li> Difficulty in Code Maintenance: inconsistent naming leads to an inconsistent codebase. The code is difficult to understand, and making changes
  feels like refactoring constantly, as you face different naming methods. Ultimately, it makes the codebase harder to maintain. </li>
</ul>
<p>In summary, not adhering to a naming convention for local variables and function parameters can lead to confusion, errors, and inefficiencies,
making the code harder to read, understand, and maintain.</p>
<h2>How to fix it</h2>
<p>First, familiarize yourself with the particular naming convention of the project in question. Then, update the name to match the convention, as
well as all usages of the name. For many IDEs, you can use built-in renaming and refactoring features to update all usages at once.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>With the default regular expression <code>^`?[_a-z][a-zA-Z0-9]*`?$</code>:</p>

```kotlin
fun printSomething(
    IMPORTANT_PARAM: String // Noncompliant
) {
    val LOCAL = "" // Noncompliant
    println(IMPORTANT_PARAM + LOCAL)
}
```
<h4>Compliant solution</h4>

```kotlin
fun printSomething(
    importantParam: String
) {
    val local = ""
    println(importantParam + local)
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Kotlin - <a href="https://kotlinlang.org/docs/coding-conventions.html#naming-rules">Coding conventions: naming rules</a> </li>
  <li> Wikipedia - <a href="https://en.wikipedia.org/wiki/Naming_convention_(programming)">Naming Convention (programming)</a> </li>
</ul>
<h3>Related rules</h3>
<ul>
  <li> <a href='checks/checks/?languages=KOTLIN&tools=SONAR_LINT&rule=kotlin:S100'>kotlin:S100</a> - Function names should comply with a naming convention </li>
  <li> <a href='checks/checks/?languages=KOTLIN&tools=SONAR_LINT&rule=kotlin:S101'>kotlin:S101</a> - Class names should comply with a naming convention </li>
</ul>