<p>This vulnerability allows attackers to impersonate a trusted host.</p>
<h2>Why is this an issue?</h2>
<p>Transport Layer Security (TLS) provides secure communication between systems over the internet by encrypting the data sent between them. In this
process, the role of hostname validation, combined with certificate validation, is to ensure that a system is indeed the one it claims to be, adding
an extra layer of trust and security.</p>
<p>When hostname validation is disabled, the client skips this critical check. This creates an opportunity for attackers to pose as a trusted entity
and intercept, manipulate, or steal the data being transmitted.</p>
<p>To do so, an attacker would obtain a valid certificate authenticating <code>example.com</code>, serve it using a different hostname, and the
application code would still accept it.</p>
<h3>What is the potential impact?</h3>
<p>Establishing trust in a secure way is a non-trivial task. When you disable hostname validation, you are removing a key mechanism designed to build
this trust in internet communication, opening your system up to a number of potential threats.</p>
<h4>Identity spoofing</h4>
<p>If a system does not validate hostnames, it cannot confirm the identity of the other party involved in the communication. An attacker can exploit
this by creating a fake server and masquerading it as a legitimate one. For example, they might set up a server that looks like your bank’s server,
tricking your system into thinking it is communicating with the bank. This scenario, called identity spoofing, allows the attacker to collect any data
your system sends to them, potentially leading to significant data breaches.</p>
<h2>How to fix it in cURL</h2>
<h3>Code examples</h3>
<p>The following code contains examples of disabled hostname validation.</p>
<p>The hostname validation gets disabled by setting <code>CURLOPT_SSL_VERIFYHOST</code> to <code>0 or false</code>. To enable validation set the value
to <code>2 or true</code> or do not set <code>CURLOPT_SSL_VERIFYHOST</code> at all to use the secure default value.</p>
<h4>Noncompliant code example</h4>

```php
$curl = curl_init();
curl_setopt($curl, CURLOPT_URL, 'https://example.com/');
curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 0);  // Noncompliant
curl_exec($curl);
curl_close($curl);
```
<h4>Compliant solution</h4>

```php
$curl = curl_init();
curl_setopt($curl, CURLOPT_URL, 'https://example.com/');
curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 2);
curl_exec($curl);
curl_close($curl);
```
<h3>How does this work?</h3>
<p>To fix the vulnerability of disabled hostname validation, it is strongly recommended to first re-enable the default validation and fix the root
cause: the validity of the certificate.</p>
<h4>Use valid certificates</h4>
<p>If a hostname validation failure prevents connecting to the target server, keep in mind that <strong>one system’s code should not work around
another system’s problems</strong>, as this creates unnecessary dependencies and can lead to reliability issues.</p>
<p>Therefore, the first solution is to change the remote host’s certificate to match its identity. If the remote host is not under your control,
consider replicating its service to a server whose certificate you can change yourself.</p>
<p>In case the contacted host is located on a development machine, and if there is no other choice, try following this solution:</p>
<ul>
  <li> Create a self-signed certificate for that machine. </li>
  <li> Add this self-signed certificate to the system’s trust store. </li>
  <li> If the hostname is not <code>localhost</code>, add the hostname in the <code>/etc/hosts</code> file. </li>
</ul>
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">Top 10 2021 Category A2 - Cryptographic Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/Top10/A05_2021-Security_Misconfiguration/">Top 10 2021 Category A5 - Security Misconfiguration</a> </li>
  <li> OWASP - <a href="https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/">Top 10 2021 Category A7 - Identification and
  Authentication Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration">Top 10 2017 Category A6 - Security
  Misconfiguration</a> </li>
  <li> OWASP - <a href="https://mas.owasp.org/checklists/MASVS-NETWORK/">Mobile AppSec Verification Standard - Network Communication Requirements</a>
  </li>
  <li> OWASP - <a href="https://owasp.org/www-project-mobile-top-10/2016-risks/m3-insecure-communication">Mobile Top 10 2016 Category M3 - Insecure
  Communication</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/297">CWE-297 - Improper Validation of Certificate with Host Mismatch</a> </li>
  <li> STIG Viewer - <a href="https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222550">Application Security and
  Development: V-222550</a> - The application must validate certificates by constructing a certification path to an accepted trust anchor. </li>
</ul>