<p>When granting users access to resources of an application, such an authorization should be based on strong decisions. For instance, a user may be
authorized to access a resource only if they are authenticated, or if they have the correct role and privileges.</p>
<h2>Why is this an issue?</h2>
<p>Access control is a critical aspect of web frameworks that ensures proper authorization and restricts access to sensitive resources or actions. To
enable access control, web frameworks offer components that are responsible for evaluating user permissions and making access control decisions. They
might examine the user’s credentials, such as roles or privileges, and compare them against predefined rules or policies to determine whether the user
should be granted access to a specific resource or action.</p>
<p>Conventionally, these checks should never grant access to every request received. If an endpoint or component is meant to be public, then it should
be ignored by access control components. Conversely, if an endpoint should deny some users from accessing it, then access control has to be configured
correctly for this endpoint.</p>
<p>Granting unrestricted access to all users can lead to security vulnerabilities and potential misuse of critical functionalities. It is important to
carefully assess access decisions based on factors such as user roles, resource sensitivity, and business requirements. Implementing a robust and
granular access control mechanism is crucial for the security and integrity of the web application itself and its surrounding environment.</p>
<h3>What is the potential impact?</h3>
<p>Not verifying user access strictly can introduce significant security risks. Some of the most prominent risks are listed below. Depending on the
use case, it is very likely that other risks are introduced on top of the ones listed.</p>
<h4>Unauthorized access</h4>
<p>As the access of users is not checked strictly, it becomes very easy for an attacker to gain access to restricted areas or functionalities,
potentially compromising the confidentiality, integrity, and availability of sensitive resources. They may exploit this access to perform malicious
actions, such as modifying or deleting data, impersonating legitimate users, or gaining administrative privileges, ultimately compromising the
security of the system.</p>
<h4>Theft of sensitive data</h4>
<p>Theft of sensitive data can result from incorrect access control if attackers manage to gain access to databases, file systems, or other storage
mechanisms where sensitive data is stored. This can lead to the theft of personally identifiable information (PII), financial data, intellectual
property, or other confidential information. The stolen data can be used for various malicious purposes, such as identity theft, financial fraud, or
selling the data on the black market, causing significant harm to individuals and organizations affected by the breach.</p>
<h2>How to fix it in Symfony</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>The <code>vote</code> method of a <a href="https://symfony.com/doc/current/security/voters.html">VoterInterface</a> implementation is not compliant
when it returns only an affirmative decision (<code>ACCESS_GRANTED</code>):</p>

```php
class NoncompliantVoter implements VoterInterface
{
    public function vote(TokenInterface $token, $subject, array $attributes)
    {
        return self::ACCESS_GRANTED; // Noncompliant
    }
}
```
<p>The <code>voteOnAttribute</code> method of a <a href="https://symfony.com/doc/current/security/voters.html">Voter</a> class is not compliant when
it returns only an affirmative decision (<code>true</code>):</p>

```php
class NoncompliantVoter extends Voter
{
    protected function supports(string $attribute, $subject)
    {
        return true;
    }

    protected function voteOnAttribute(string $attribute, $subject, TokenInterface $token)
    {
        return true; // Noncompliant
    }
}
```
<h4>Compliant solution</h4>
<p>The <code>vote</code> method of a <a href="https://symfony.com/doc/current/security/voters.html">VoterInterface</a> type should return a negative
decision (<code>ACCESS_DENIED</code>) or abstain from making a decision (<code>ACCESS_ABSTAIN</code>):</p>

```php
class CompliantVoter implements VoterInterface
{
    public function vote(TokenInterface $token, $subject, array $attributes)
    {
        if (foo()) {
            return self::ACCESS_GRANTED;
        } else if (bar()) {
            return self::ACCESS_ABSTAIN;
        }
        return self::ACCESS_DENIED;
    }
}
```
<p>The <code>voteOnAttribute</code> method of a <a href="https://symfony.com/doc/current/security/voters.html">Voter</a> type should return a negative
decision (<code>false</code>):</p>

```php
class CompliantVoter extends Voter
{
    protected function supports(string $attribute, $subject)
    {
        return true;
    }

    protected function voteOnAttribute(string $attribute, $subject, TokenInterface $token)
    {
        if (foo()) {
            return true;
        }
        return false;
    }
}
```
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A01_2021-Broken_Access_Control/">Top 10 2021 Category A1 - Broken Access Control</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A5_2017-Broken_Access_Control">Top 10 2017 Category A5 - Broken Access Control</a>
  </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/285">CWE-285 - Improper Authorization</a> </li>
</ul>