Using standard C input/output functions can introduce risks of undefined or implementation-dependent behavior in your code. 
It is generally recommended to use safer and more robust I/O libraries or dedicated logging mechanisms appropriate for your environment and platform.

# What Does This Check Look For?
This check reports a finding when it detects the use of C-style standard input/output functions declared in the `<stdio.h>` or `<wchar.h>` header files.
This includes functions like `printf`, `scanf`, `fopen`, `fwrite`, `fwprintf`, `fwscanf`, etc.

# Why is This a Problem?
The standard C I/O functions, while widely known, can be a source of issues in modern software development, especially embedded contexts or C++. 
Undefined or implementation-dependent behavior is a primary concern. 
For example:
* Using functions like `printf`, `fwprintf`, or `vfprintf` for logging presents limited structure and control since they lack log severity levels.
In multithreaded programs, those calls can lead to interleaved and unreadable output if not externally synchronized.
* Error handling with these functions (like checking `ferror` or `feof`) can be error-prone and easily overlooked, leading to unexpected program states. 
* Buffer overflows are another common risk, particularly with functions like `gets` (which is deprecated for this reason) or improper use of `scanf` without specifying field widths. 
* Furthermore, in C++ programs, mixing C-style I/O with C++ iostreams can lead to synchronization issues and other subtle bugs if not handled carefully. 
* Using these functions might also limit portability or compatibility in certain specialized environments.

# How Can I Resolve This?
This check covers a broad set of functions. So, depending on your context and which function(s) this involves, you may consider:
* A Dedicated Logging Framework: For diagnostic messages or system event logging, consider using a dedicated logging library or framework, instead of direct calls to functions like `printf` or `fprintf`. 
Logging frameworks offer more flexibility (e.g., log levels, different output destinations, runtime configuration, structured logging), better performance in some scenarios, and abstract away the raw I/O calls.
* Alternative C libraries designed for safer or more specialized I/O operations if available and suitable for your project (e.g., Sfio - Safe/Fast I/O Library).
* For certain applications (e.g., embedded systems, low-level utilities), it might be more appropriate to use operating system-specific I/O system calls directly (e.g., POSIX open, read, write, close on Unix-like systems).
Specialized libraries can offer enhanced safety or performance. 
OS-specific APIs provide direct control and can be necessary in constrained environments, bypassing the stdio buffering and formatting layers which might be the source of issues.
* Use C++ Iostreams (if available): If you are working in a C++ environment, the `<iostream>` library offers a more type-safe approach to I/O. 
It often integrates better with other C++ features and can provide better error handling mechanisms through exceptions.
* Careful use and justification (if unavoidable): If using standard C I/O functions is absolutely necessary, ensure that their use is meticulously reviewed for safety (e.g., always checking return values, ensuring proper error handling with `errno`, `ferror`, `feof`, using `width` specifiers with `scanf`, and secure versions like `snprintf`). 
The reasons for their use should be clearly documented. 
This approach should generally be a last resort and might involve suppressing the check with clear justification.

# Examples

## Non-Compliant
```c
#include <stdio.h>

void log_std(const char* message) {
    FILE* logfile = fopen("app_stdio.log", "a"); // non-compliant: uses fopen
    if (logfile != NULL) {
        fprintf(logfile, "%s\n", message);    // non-compliant: uses fprintf
        fclose(logfile); // non-compliant: as a consequence of using fopen
    }
}

int main() {
    printf("Initializing via stdio...\n"); // non-compliant: uses printf
    log_std("Application started using stdio logging.");
    return 0;
}
```

## Compliant
```c
#include <unistd.h> 
#include <string.h>
#include <syslog.h>

void app_log_message(int priority, const char* message) {
    openlog("my_app_name", LOG_PID | LOG_CONS, LOG_USER);
    syslog(priority, "%s", message); // compliant: using syslog instead of fprintf or printf
    closelog();
}

int main() {
    const char* init_msg = "Initializing application (direct to stdout via POSIX write)...\n";
    write(STDOUT_FILENO, init_msg, strlen(init_msg)); // compliant: POSIX write
    app_log_message(LOG_INFO, "Application started successfully.");
    
    // Example of an error log message
    // app_log_message(LOG_ERR, "A simulated error occurred during processing.");
    
    return 0;
}
```

## Compliant
```cpp
#include <iostream>
#include <fstream>
#include <string>

void log_cpp(const std::string& message) {
    std::ofstream logfile("app_cpp.log", std::ios_base::app); // compliant: C++ iostreams
    if (logfile.is_open()) {
        logfile << message << std::endl;
    }
}

int main() {
    std::cout << "Initializing via iostream..." << std::endl; // compliant: uses std::cout
    log_cpp("Application started using C++ iostream logging.");
    return 0;
}
```

# Where Can I Learn More?
- [C++ Code Guidelines. SL.io.3: Prefer iostreams for I/O](https://isocpp.github.io/CppCoreGuidelines/CppCoreGuidelines#slio3-prefer-iostreams-for-io)
- [CERT C Coding Standard: FIO30-C - Exclude user input from format strings (Related to security aspects of printf/scanf family)](https://wiki.sei.cmu.edu/confluence/display/c/FIO30-C.+Exclude+user+input+from+format+strings)
- [CERT C Coding Standard: FIO34-C - Distinguish between characters read from a file and EOF or WEOF](https://wiki.sei.cmu.edu/confluence/display/c/FIO34-C.+Distinguish+between+characters+read+from+a+file+and+EOF+or+WEOF)
- [CERT C Coding Standard: FIO37-C - Do not assume that fgets() or fgetws() returns a nonempty string when successful](https://wiki.sei.cmu.edu/confluence/display/c/FIO37-C.+Do+not+assume+that+fgets%28%29+or+fgetws%28%29+returns+a+nonempty+string+when+successful)
- [CERT C Coding Standard: FIO40-C - Reset strings on fgets() or fgetws() failure](https://wiki.sei.cmu.edu/confluence/display/c/FIO40-C.+Reset+strings+on+fgets%28%29++or+fgetws%28%29+failure)
- [CERT C Coding Standard: FIO41-C - Do not call getc(), putc(), getwc(), or putwc() with a stream argument that has side effects](https://wiki.sei.cmu.edu/confluence/display/c/FIO41-C.+Do+not+call+getc%28%29%2C+putc%28%29%2C+getwc%28%29%2C+or+putwc%28%29+with+a+stream+argument+that+has+side+effects)
- [CERT C Coding Standard: FIO47-C - Use valid format strings](https://wiki.sei.cmu.edu/confluence/display/c/FIO47-C.+Use+valid+format+strings)