Declaring an `OUT` parameter as `IN OUT` can lead to unexpected behavior and potential bugs. 
This is because the parameter is declared as an output only, but the `IN OUT` mode implies that it can also be used as an input. 
This can cause confusion for developers who are not aware of the parameter's true purpose, and may lead to incorrect code being written.

This check identifies routines that declare an `IN OUT` parameter but treat it as `OUT`.

# Why is This a Problem?
Declaring an `OUT` parameter as `IN OUT` can lead to several problems, including:

* Incorrect usage: Developers may mistakenly assume that an `OUT` parameter declared as `IN OUT` can be used as an input, leading to incorrect assignments and unexpected behavior.
* Difficulty tracking changes: Since changes made to an `OUT` parameter declared as `IN OUT` are reflected in the caller's scope, it can be difficult to track the validity and consistency of the data being passed around.
* Potential for confusion: The inconsistent usage of the parameter can lead to confusion and bugs, especially when code is reviewed or modified by multiple developers.

# How Can I Resolve This?
To avoid declaring an `OUT` parameter as `IN OUT`, follow these guidelines:

* Understand the parameter's usage: Clearly define the intended purpose of each parameter before declaring it. If a parameter is solely for output, declare it as OUT only.
* Use the correct mode for each parameter: Ensure that the mode of each parameter accurately reflects its intended usage.
* Review code carefully: When reviewing code, carefully examine the usage of `OUT` parameters to ensure that they are being declared and used consistently.

## Example (before)
```oraclesqlplus
DECLARE
    PROCEDURE sample_procedure(p_parameter IN OUT VARCHAR2) IS
    BEGIN
        -- ignore the input value and assign a new value
        p_parameter := 'New value';
    END sample_procedure;
```

## Example (after)
```oraclesqlplus
DECLARE
    PROCEDURE sample_procedure(p_parameter OUT VARCHAR2) IS
    BEGIN
        p_parameter := 'New value';
    END sample_procedure;
```

# Where Can I Learn More?

- [G-7170: Avoid using an IN OUT parameter as IN or OUT only](https://trivadis.github.io/plsql-and-sql-coding-guidelines/v4.2/4-language-usage/7-stored-objects/1-general/g-7170/)


