<h2>Why is this an issue?</h2>
<p><code>Cloneable</code> is the marker <code>Interface</code> that indicates that <code>clone()</code> may be called on an object. Overriding
<code>clone()</code> without implementing <code>Cloneable</code> can be helpful if you want to control how subclasses clone themselves, but otherwise,
it’s probably a mistake.</p>
<p>The usual convention for <code>Object.clone()</code> according to Oracle’s Javadoc is:</p>
<ol>
  <li> <code>x.clone() != x</code> </li>
  <li> <code>x.clone().getClass() == x.getClass()</code> </li>
  <li> <code>x.clone().equals(x)</code> </li>
</ol>
<p>Obtaining the object that will be returned by calling <code>super.clone()</code> helps to satisfy those invariants:</p>
<ol>
  <li> <code>super.clone()</code> returns a new object instance </li>
  <li> <code>super.clone()</code> returns an object of the same type as the one <code>clone()</code> was called on </li>
  <li> <code>Object.clone()</code> performs a shallow copy of the object’s state. </li>
</ol>
<h2>How to fix it</h2>
<p>Ensure that the <code>clone()</code> method calls <code>super.clone()</code> and implement <code>Cloneable</code> in the class or remove the clone
method.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
class BaseClass {  // Noncompliant - should implement Cloneable
  @Override
  public Object clone() throws CloneNotSupportedException {    // Noncompliant - should return the super.clone() instance
    return new BaseClass();
  }
}

class DerivedClass extends BaseClass implements Cloneable {
  /* Does not override clone() */

  public void sayHello() {
    System.out.println("Hello, world!");
  }
}

class Application {
  public static void main(String[] args) throws Exception {
    DerivedClass instance = new DerivedClass();
    ((DerivedClass) instance.clone()).sayHello();              // Throws a ClassCastException because invariant #2 is violated
  }
}
```
<h4>Compliant solution</h4>

```java
class BaseClass implements Cloneable {
  @Override
  public Object clone() throws CloneNotSupportedException {    // Compliant
    return super.clone();
  }
}

class DerivedClass extends BaseClass implements Cloneable {
  /* Does not override clone() */

  public void sayHello() {
    System.out.println("Hello, world!");
  }
}

class Application {
  public static void main(String[] args) throws Exception {
    DerivedClass instance = new DerivedClass();
    ((DerivedClass) instance.clone()).sayHello();              // Displays "Hello, world!" as expected. Invariant #2 is satisfied
  }
}
```
<h2>Resources</h2>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/580">CWE-580 - clone() Method Without super.clone()</a> </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/FjZGBQ">CERT, MET53-J.</a> - Ensure that the clone() method calls super.clone() </li>
</ul>