/*
 * Decompiled with CFR 0.152.
 */
package com.teamscale.index.gitbridge;

import com.teamscale.index.gitbridge.GitBridgeException;
import com.teamscale.index.gitbridge.ZipToGitImporterUtils;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import org.conqat.lib.commons.assertion.CCSMAssert;
import org.conqat.lib.commons.collections.CollectionUtils;
import org.conqat.lib.commons.filesystem.AntPatternDirectoryScanner;
import org.conqat.lib.commons.filesystem.CanonicalFile;
import org.conqat.lib.commons.filesystem.FileSystemUtils;
import org.conqat.lib.commons.filesystem.ZipFile;
import org.conqat.lib.commons.string.StringUtils;
import org.eclipse.jgit.api.AddCommand;
import org.eclipse.jgit.api.CommitCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.ResetCommand;
import org.eclipse.jgit.api.RmCommand;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.api.errors.NoHeadException;
import org.eclipse.jgit.diff.DiffEntry;
import org.eclipse.jgit.errors.NoWorkTreeException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.NullProgressMonitor;
import org.eclipse.jgit.lib.ObjectReader;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.ProgressMonitor;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevTree;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import org.eclipse.jgit.treewalk.AbstractTreeIterator;
import org.eclipse.jgit.treewalk.CanonicalTreeParser;

public class ZipToGitImporter
implements AutoCloseable {
    private static final String[] DOT_GIT_EXCLUDE = new String[]{".git/**"};
    private final Git git;
    private String resetRevisionCommitName;
    private boolean repositoryIsInInitialState;
    private Function<String, String> pathTransformation = Function.identity();
    private static final int MAX_TRIES_TO_DELETE_WORK_TREE_CONTENT = 3;
    private static final int WAIT_SECONDS_BETWEEN_TRIES_TO_DELETE_WORK_TREE = 2;

    public ZipToGitImporter(CanonicalFile gitRepositoryLocation) throws GitBridgeException {
        this.git = ZipToGitImporter.initGit((File)gitRepositoryLocation);
        this.useHeadAsResetRevision();
        this.repositoryIsInInitialState = this.getLastCommit() == null;
    }

    public void setPathTransformation(Function<String, String> pathTransformation) {
        this.pathTransformation = pathTransformation;
    }

    private static Git initGit(File gitRepositoryLocation) throws GitBridgeException {
        File repositoryFolder = null;
        try {
            if (!ZipToGitImporterUtils.isExisitingRepository(gitRepositoryLocation)) {
                Git.init().setInitialBranch("master").setDirectory(gitRepositoryLocation).call();
            }
            repositoryFolder = new File(gitRepositoryLocation.getAbsoluteFile(), ".git");
            Repository repository = FileRepositoryBuilder.create((File)repositoryFolder);
            Git git = new Git(repository);
            StoredConfig config = git.getRepository().getConfig();
            config.setBoolean("core", null, "longpaths", true);
            config.save();
            return git;
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Unable to initialize git repository at " + String.valueOf(gitRepositoryLocation), e);
        }
        catch (IOException e) {
            throw new GitBridgeException("Can not access repository " + String.valueOf(repositoryFolder), e);
        }
    }

    @Override
    public void close() {
        this.git.close();
    }

    public boolean commit(String committerName, String committerMail, Instant commitTime, String message, boolean commitAll) throws GitBridgeException {
        try {
            if (!this.git.status().call().hasUncommittedChanges()) {
                return false;
            }
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Error during retrieval of git status.", e);
        }
        CommitCommand commit = this.git.commit();
        commit.setCommitter(new PersonIdent(committerName, committerMail, Date.from(commitTime), TimeZone.getDefault()));
        commit.setMessage(message);
        commit.setAll(commitAll);
        try {
            commit.call();
            this.repositoryIsInInitialState = false;
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Error during git commit.", e);
        }
        return true;
    }

    public boolean add(List<String> pathsToAdd) throws GitBridgeException {
        if (pathsToAdd.isEmpty()) {
            return false;
        }
        AddCommand add = this.git.add();
        for (String path : pathsToAdd) {
            add.addFilepattern(ZipToGitImporter.cleanPath(path));
        }
        try {
            add.call();
            Status statusCommand = this.git.status().call();
            ArrayList addedAndModifiedFiles = new ArrayList(statusCommand.getAdded());
            addedAndModifiedFiles.addAll(statusCommand.getChanged());
            return !addedAndModifiedFiles.isEmpty();
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Error during adding zip content: Can't add content to working tree " + String.valueOf(this.getWorkTree()), e);
        }
    }

    public List<String> extractZipToWorkTree(ZipFile zip, String entryPrefix) throws GitBridgeException {
        try {
            List<String> extractedPaths = FileSystemUtils.unzip((ZipFile)zip, (File)this.getWorkTree(), (String)entryPrefix);
            extractedPaths = this.applyTransformation(extractedPaths);
            return extractedPaths;
        }
        catch (IOException e) {
            throw new GitBridgeException("Error during extracting zip content: Can't unzip " + String.valueOf(zip) + " to " + String.valueOf(this.getWorkTree()), e);
        }
    }

    private List<String> applyTransformation(List<String> paths) throws GitBridgeException {
        ArrayList<String> transformedPaths = new ArrayList<String>();
        for (String path : paths) {
            String transformedPath = this.pathTransformation.apply(path);
            if (!transformedPath.equals(path)) {
                this.move(path, transformedPath, false);
            }
            transformedPaths.add(transformedPath);
        }
        return transformedPaths;
    }

    public File getWorkTree() {
        return this.git.getRepository().getWorkTree();
    }

    public List<String> delete(List<String> filePaths, boolean ignoreMissing) throws GitBridgeException {
        ArrayList<String> pathsUnableToDelete = new ArrayList<String>();
        ArrayList<String> pathsForGitRm = new ArrayList<String>();
        RmCommand rm = this.git.rm();
        for (String filePath : filePaths) {
            String cleanFilePath = ZipToGitImporter.cleanPath(filePath);
            File file = new File(this.getWorkTree(), cleanFilePath);
            if (ignoreMissing && !file.exists()) continue;
            if (!file.isFile()) {
                pathsUnableToDelete.add(cleanFilePath);
                this.removeWorkTreeDirectoryIfEmpty(cleanFilePath);
                continue;
            }
            rm = rm.addFilepattern(cleanFilePath);
            pathsForGitRm.add(cleanFilePath);
        }
        if (pathsForGitRm.isEmpty()) {
            return pathsUnableToDelete;
        }
        try {
            rm.call();
        }
        catch (GitAPIException e) {
            try {
                throw new GitBridgeException("Error when deleting files from git", e);
            }
            catch (Throwable throwable) {
                for (String filePath : pathsForGitRm) {
                    this.removeWorkTreeDirectoryIfEmpty(filePath);
                }
                throw throwable;
            }
        }
        for (String filePath : pathsForGitRm) {
            this.removeWorkTreeDirectoryIfEmpty(filePath);
        }
        return pathsUnableToDelete;
    }

    public boolean isFile(String filePath) throws GitBridgeException {
        String cleanFilePath = ZipToGitImporter.cleanPath(filePath);
        return new File(this.getWorkTree(), cleanFilePath).isFile();
    }

    public boolean isInitialRepository() {
        return this.repositoryIsInInitialState;
    }

    public String[] findPathsInWorkTree(String ... includePatterns) throws GitBridgeException {
        String[] matchingFiles;
        try {
            matchingFiles = AntPatternDirectoryScanner.scan((String)this.getWorkTree().getPath(), (boolean)true, (String[])includePatterns, (String[])DOT_GIT_EXCLUDE);
        }
        catch (IOException e) {
            throw new GitBridgeException("Can't scan working tree at " + this.getWorkTree().getPath(), e);
        }
        for (int i = 0; i < matchingFiles.length; ++i) {
            matchingFiles[i] = FileSystemUtils.normalizeSeparators((String)matchingFiles[i]);
        }
        return matchingFiles;
    }

    public void move(String fromPath, String toPath, boolean ignoreMissingSource) throws GitBridgeException {
        File source = new File(this.getWorkTree(), fromPath);
        if (ignoreMissingSource && !source.exists()) {
            return;
        }
        File destination = new File(this.getWorkTree(), toPath);
        try {
            if (source.getCanonicalPath().equals(destination.getCanonicalPath())) {
                return;
            }
            FileSystemUtils.ensureParentDirectoryExists((File)destination);
            FileSystemUtils.deleteFile((File)destination);
            FileSystemUtils.renameFileTo((File)source, (File)destination);
        }
        catch (IOException e) {
            throw new GitBridgeException("Can't move work tree path from " + String.valueOf(source) + " to " + String.valueOf(destination), e);
        }
        try {
            this.git.rm().addFilepattern(ZipToGitImporter.cleanPath(fromPath)).call();
            this.removeWorkTreeDirectoryIfEmpty(fromPath);
            this.git.add().addFilepattern(ZipToGitImporter.cleanPath(toPath)).call();
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Error when performing git move from " + fromPath + " to " + toPath, e);
        }
    }

    public Instant getLastCommitTime() throws GitBridgeException {
        RevCommit lastCommit = this.getLastCommit();
        if (lastCommit == null) {
            throw new GitBridgeException("Unable to get time of last commit since no commit exists.");
        }
        return lastCommit.getCommitterIdent().getWhen().toInstant();
    }

    private RevCommit getLastCommit() throws GitBridgeException {
        Iterable commitLog;
        try {
            commitLog = this.git.log().setMaxCount(1).call();
        }
        catch (NoHeadException e) {
            return null;
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Unable to retrieve last commit", e);
        }
        return (RevCommit)CollectionUtils.getAny((Iterable)commitLog);
    }

    private void removeWorkTreeDirectoryIfEmpty(String path) throws GitBridgeException {
        try {
            FileSystemUtils.recursivelyRemoveDirectoryIfEmpty((File)new File(this.getWorkTree(), path));
        }
        catch (IOException e) {
            throw new GitBridgeException("Unable to recursively delete empty directory " + path, e);
        }
    }

    private static String cleanPath(String path) throws GitBridgeException {
        if (path == null) {
            throw new GitBridgeException("Unexpected null path when updating Git repository.");
        }
        return path.replaceAll("//+", "/");
    }

    public void checkCleanWorkTree(String message) throws GitBridgeException {
        Status status;
        try {
            status = this.git.status().call();
        }
        catch (GitAPIException | NoWorkTreeException e) {
            throw new GitBridgeException("Unable to get git status", e);
        }
        ZipToGitImporter.checkEmptyStatusSet("added", status.getAdded(), message);
        ZipToGitImporter.checkEmptyStatusSet("changed", status.getChanged(), message);
        ZipToGitImporter.checkEmptyStatusSet("conflicting", status.getConflicting(), message);
        ZipToGitImporter.checkEmptyStatusSet("ignored not in index", status.getIgnoredNotInIndex(), message);
        ZipToGitImporter.checkEmptyStatusSet("missing", status.getMissing(), message);
        ZipToGitImporter.checkEmptyStatusSet("modified", status.getModified(), message);
        ZipToGitImporter.checkEmptyStatusSet("removed", status.getRemoved(), message);
        ZipToGitImporter.checkEmptyStatusSet("uncommitted changes", status.getUncommittedChanges(), message);
        ZipToGitImporter.checkEmptyStatusSet("untracked", status.getUntracked(), message);
        ZipToGitImporter.checkEmptyStatusSet("untracked folders", status.getUntrackedFolders(), message);
        ZipToGitImporter.checkEmptyStatusSet("conflicting stage state", status.getConflictingStageState().keySet(), message);
    }

    private static void checkEmptyStatusSet(String statusSetName, Set<String> statusSet, String message) throws GitBridgeException {
        if (!statusSet.isEmpty()) {
            throw new GitBridgeException("Dirty work tree after commit (" + message + ") " + statusSetName + " = " + String.valueOf(statusSet));
        }
    }

    public void resetGit() throws GitBridgeException {
        if (this.resetRevisionCommitName == null) {
            File repositoryLocation = this.getWorkTree();
            try {
                FileSystemUtils.deleteRecursively((File)repositoryLocation);
            }
            catch (IOException e) {
                throw new GitBridgeException(e);
            }
            if (repositoryLocation.exists()) {
                throw new GitBridgeException("Unable to clear git repository directory at reset to initial.");
            }
            ZipToGitImporter.initGit(repositoryLocation);
        }
        ResetCommand reset = this.git.reset();
        reset.setRef(this.resetRevisionCommitName).setMode(ResetCommand.ResetType.HARD);
        try {
            reset.call();
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Unable to perform Git reset to revision " + this.resetRevisionCommitName, e);
        }
    }

    public void useHeadAsResetRevision() throws GitBridgeException {
        RevCommit lastCommit = this.getLastCommit();
        this.resetRevisionCommitName = lastCommit != null ? lastCommit.getName() : null;
    }

    public void deleteWorkTreeContent(Collection<String> foldersToPreserve) throws GitBridgeException {
        this.deleteWorkTreeContent(CollectionUtils.unionSet(foldersToPreserve, (Collection[])new Collection[]{List.of(".git")}), 3);
    }

    private void deleteWorkTreeContent(Set<String> foldersToPreserve, int maxTries) throws GitBridgeException {
        Object[] workTreeChildren = this.getWorkTree().list();
        CCSMAssert.isNotNull((Object)workTreeChildren, (String)"The work tree must be a directory once we arrive here.");
        foldersToPreserve = CollectionUtils.mapToSet(foldersToPreserve, folderName -> StringUtils.stripSuffix((String)folderName, (String)"/"));
        HashSet childrenToDelete = CollectionUtils.asHashSet((Object[])workTreeChildren);
        childrenToDelete.removeAll(foldersToPreserve);
        for (String fileToDelete : childrenToDelete) {
            File file = new File(this.getWorkTree(), fileToDelete);
            try {
                FileSystemUtils.deleteRecursively((File)file);
            }
            catch (IOException e) {
                throw new GitBridgeException(e);
            }
        }
        if (!this.isEmptyWorkTree(foldersToPreserve)) {
            this.tryDeleteWorkTreeContents(foldersToPreserve, maxTries, (String[])workTreeChildren);
        }
    }

    private void tryDeleteWorkTreeContents(Set<String> foldersToPreserve, int maxTries, String[] workTreeChildren) throws GitBridgeException {
        if (maxTries > 1) {
            try {
                TimeUnit.SECONDS.sleep(2L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        } else {
            throw new GitBridgeException("Unexpected files/directories in work tree after emptying: " + Arrays.toString(workTreeChildren));
        }
        this.deleteWorkTreeContent(foldersToPreserve, maxTries - 1);
    }

    public IntroducedGitChanges getGitChangesFromSync() throws GitBridgeException {
        try {
            Status status = this.git.status().call();
            ArrayList<String> addedAndModified = new ArrayList<String>();
            addedAndModified.addAll(status.getAdded());
            addedAndModified.addAll(status.getModified());
            addedAndModified.addAll(status.getChanged());
            return new IntroducedGitChanges(addedAndModified, status.getRemoved().stream().toList(), Map.of(), false);
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Could not get introduced changes to git repository. " + e.getMessage(), e);
        }
    }

    public void update() throws GitBridgeException {
        try {
            this.git.add().addFilepattern(".").setUpdate(true).call();
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Unable to update index.", e);
        }
    }

    private boolean isEmptyWorkTree(Set<String> allowedFolders) {
        String[] children = this.getWorkTree().list();
        if (children == null) {
            return true;
        }
        for (String child : children) {
            if (allowedFolders.contains(child)) continue;
            return false;
        }
        return true;
    }

    public Set<String> getChangedFilesSince(Instant changesSince) throws GitBridgeException {
        HashSet<String> changedFiles = new HashSet<String>();
        try {
            RevCommit mostRecentCommit = this.getLastCommit();
            RevCommit commitBeforeStart = this.getFirstCommitBefore(changesSince);
            CCSMAssert.isNotNull((Object)commitBeforeStart, (String)("There is already an initial commit to the abap git before or at the timestamp of the current import (" + String.valueOf(changesSince) + "). Aborting current import."));
            List diffs = this.git.diff().setOldTree(this.createTreeIterator(commitBeforeStart)).setNewTree(this.createTreeIterator(mostRecentCommit)).call();
            for (DiffEntry diff : diffs) {
                if (diff.getChangeType() == DiffEntry.ChangeType.DELETE) continue;
                changedFiles.add(diff.getNewPath());
            }
        }
        catch (IOException | GitAPIException e) {
            throw new GitBridgeException("Error during getting changed files", e);
        }
        return changedFiles;
    }

    private AbstractTreeIterator createTreeIterator(RevCommit commit) throws IOException {
        try (RevWalk walk = new RevWalk(this.git.getRepository());){
            RevTree tree = walk.parseTree((AnyObjectId)commit.getTree().getId());
            CanonicalTreeParser treeParser = new CanonicalTreeParser();
            try (ObjectReader reader = this.git.getRepository().newObjectReader();){
                treeParser.reset(reader, (AnyObjectId)tree.getId());
            }
            walk.dispose();
            CanonicalTreeParser canonicalTreeParser = treeParser;
            return canonicalTreeParser;
        }
    }

    private RevCommit getFirstCommitBefore(Instant time) throws GitAPIException {
        Iterable commits = this.git.log().call();
        for (RevCommit commit : commits) {
            Instant commitTime = commit.getCommitterIdent().getWhen().toInstant();
            if (!commitTime.isBefore(time)) continue;
            return commit;
        }
        return null;
    }

    public void replaceWithHead(List<String> paths) throws GitBridgeException {
        if (CollectionUtils.isNullOrEmpty(paths)) {
            return;
        }
        try {
            this.git.checkout().addPaths(paths).call();
        }
        catch (GitAPIException e) {
            throw new GitBridgeException("Unable to replace to HEAD", e);
        }
    }

    public String readContent(String filePath, Charset encoding) throws GitBridgeException {
        String cleanFilePath = ZipToGitImporter.cleanPath(filePath);
        File file = new File(this.getWorkTree(), cleanFilePath);
        if (!file.isFile()) {
            throw new GitBridgeException("File '" + cleanFilePath + "' which should be read does not exist in the work tree or is not a file.");
        }
        try {
            return FileSystemUtils.readFile((File)file, (Charset)encoding);
        }
        catch (IOException e) {
            throw new GitBridgeException("File '" + cleanFilePath + "' could not be read.", e);
        }
    }

    public void replaceContent(String filePath, String newContent, Charset encoding) throws GitBridgeException {
        String cleanFilePath = ZipToGitImporter.cleanPath(filePath);
        File file = new File(this.getWorkTree(), cleanFilePath);
        if (!file.isFile()) {
            throw new GitBridgeException("File '" + cleanFilePath + "' which should be written does not exist in the work tree or is not a file.");
        }
        try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(file, false), encoding);){
            writer.write(newContent);
        }
        catch (IOException e) {
            throw new GitBridgeException("File '" + cleanFilePath + "' could not be written.", e);
        }
    }

    public void runAutoGarbageCollection() {
        this.git.getRepository().autoGC((ProgressMonitor)NullProgressMonitor.INSTANCE);
    }

    public record IntroducedGitChanges(List<String> addedAndChangedFiles, List<String> deletedFiles, Map<String, String> movedFiles, boolean commitSuccessful) {
        public boolean hasAdditionsAndModifications() {
            return !this.addedAndChangedFiles.isEmpty();
        }

        public boolean hasDeletions() {
            return !this.deletedFiles.isEmpty();
        }

        public boolean hasMoves() {
            return !this.movedFiles.isEmpty();
        }

        public IntroducedGitChanges merge(IntroducedGitChanges gitChanges) {
            ArrayList<String> newFiles = new ArrayList<String>(this.addedAndChangedFiles());
            newFiles.addAll(gitChanges.addedAndChangedFiles());
            ArrayList<String> removedFiles = new ArrayList<String>(this.deletedFiles());
            removedFiles.addAll(gitChanges.deletedFiles());
            HashMap<String, String> movedFiles = new HashMap<String, String>(this.movedFiles());
            movedFiles.putAll(gitChanges.movedFiles());
            return new IntroducedGitChanges(newFiles, removedFiles, movedFiles, false);
        }
    }
}

