<p>Development tools and frameworks usually have options to make debugging easier for developers. Although these features are useful during
development, they should never be enabled for applications deployed in production. Debug instructions or error messages can leak detailed information
about the system, like the application’s path or file names.</p>
<h2>Ask Yourself Whether</h2>
<ul>
  <li> The code or configuration enabling the application debug features is deployed on production servers or distributed to end users. </li>
  <li> The application runs by default with debug features activated. </li>
</ul>
<p>There is a risk if you answered yes to any of those questions.</p>
<h2>Recommended Secure Coding Practices</h2>
<p>Do not enable debugging features on production servers.</p>
<p>The .Net Core framework offers multiple features which help during debug.
<code>Microsoft.AspNetCore.Builder.IApplicationBuilder.UseDeveloperExceptionPage</code> and
<code>Microsoft.AspNetCore.Builder.IApplicationBuilder.UseDatabaseErrorPage</code> are two of them. Make sure that those features are disabled in
production.</p>
<p>Use <code>if (env.IsDevelopment())</code> to disable debug code.</p>
<h2>Sensitive Code Example</h2>
<p>This rule raises issues when the following .Net Core methods are called:
<code>Microsoft.AspNetCore.Builder.IApplicationBuilder.UseDeveloperExceptionPage</code>,
<code>Microsoft.AspNetCore.Builder.IApplicationBuilder.UseDatabaseErrorPage</code>.</p>
<pre>
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;

namespace mvcApp
{
    public class Startup2
    {
        public void Configure(IApplicationBuilder app, IHostingEnvironment env)
        {
            // Those calls are Sensitive because it seems that they will run in production
            app.UseDeveloperExceptionPage(); // Sensitive
            app.UseDatabaseErrorPage(); // Sensitive
        }
    }
}
</pre>
<h2>Compliant Solution</h2>
<pre>
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;

namespace mvcApp
{
    public class Startup2
    {
        public void Configure(IApplicationBuilder app, IHostingEnvironment env)
        {
            if (env.IsDevelopment())
            {
                // The following calls are ok because they are disabled in production
                app.UseDeveloperExceptionPage(); // Compliant
                app.UseDatabaseErrorPage(); // Compliant
            }
        }
    }
}
</pre>
<h2>Exceptions</h2>
<p>This rule does not analyze configuration files. Make sure that debug mode is not enabled by default in those files.</p>
<h2>See</h2>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A05_2021-Security_Misconfiguration/">Top 10 2021 Category A5 - Security Misconfiguration</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/489">CWE-489 - Active Debug Code</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/215">CWE-215 - Information Exposure Through Debug Information</a> </li>
</ul>

