<h2>Why is this an issue?</h2>
<p>If an <code>InterruptedException</code> or a <code>ThreadDeath</code> error is not handled properly, the information that the thread was
interrupted will be lost. Handling this exception means either to re-throw it or manually re-interrupt the current thread by calling
<code>Thread.interrupt()</code>. Simply logging the exception is not sufficient and counts as ignoring it. Between the moment the exception is caught
and handled, is the right time to perform cleanup operations on the method’s state, if needed.</p>
<h3>What is the potential impact?</h3>
<p>Failing to interrupt the thread (or to re-throw) risks delaying the thread shutdown and losing the information that the thread was interrupted -
probably without finishing its task.</p>
<h3>Noncompliant code example</h3>

```java
public void run () {
  try {
    /*...*/
  } catch (InterruptedException e) { // Noncompliant; logging is not enough
    LOGGER.log(Level.WARN, "Interrupted!", e);
  }
}
```
<h3>Compliant solution</h3>

```java
public void run () {
  try {
    /* ... */
  } catch (InterruptedException e) { // Compliant; the interrupted state is restored
    LOGGER.log(Level.WARN, "Interrupted!", e);
    /* Clean up whatever needs to be handled before interrupting  */
    Thread.currentThread().interrupt();
  }
}

public void run () {
  try {
    /* ... */
  } catch (ThreadDeath e) { // Compliant; the error is being re-thrown
    LOGGER.log(Level.WARN, "Interrupted!", e);
    /* Clean up whatever needs to be handled before re-throwing  */
    throw e;
  }
}
```
<h2>Resources</h2>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/391">CWE-391 - Unchecked Error Condition</a> </li>
</ul>