<h2>Why is this an issue?</h2>
<p>JavaScript will automatically insert semicolons when parsing the code so invalid sequences can be "fixed" to valid syntax. This behavior, called <a
href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Lexical_grammar#automatic_semicolon_insertion">"Automatic semicolon
insertion"</a> or <strong>ASI</strong>, makes semicolons at the end of statements optional and attempts to make JavaScript more approachable and
convenient.</p>
<p>However, sometimes, relying on ASI can lead to unexpected results. ASI will only be triggered if a line break separates tokens that would otherwise
produce invalid syntax. JavaScript will not insert semicolons if the next token can be parsed as part of a valid structure.</p>
<p>In the case of function call arguments, they are allowed to be on a separate line. But, depending on the developer’s intent and, especially when
working with <a href="https://developer.mozilla.org/en-US/docs/Glossary/IIFE">IIFE</a> (or any other design pattern using <a
href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Grouping">Grouping operator</a>), it can lead to errors and most
likely <em>will</em> lead to questions for maintainers.</p>
<p>What was the initial intent of the developer?</p>
<ol>
  <li> Defining a function and then executing some unrelated code inside a closure? </li>
  <li> Passing the second function as a parameter to the first one? </li>
</ol>
<p>The first option will be the one chosen by the JavaScript interpreter.</p>

```js
const fn = function () {
  //...
}

(function () { // Noncompliant: function is passed as a parameter to fn
  //...
})();
```
<p>By extension, and to improve readability, any kind of function call argument should not start on a new line.</p>

```js
// Define a function
const fn = function () {
  //...
}; // <-- semicolon added

// then execute some code inside a closure
(function () {
  //...
})();
```
<p>Or</p>

```js
var fn = function () {
  //...
}(function () { // <-- start function call arguments on same line
  //...
})();
```
<p>Similarly, <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Template_literals#tagged_templates">tagged templates</a>
allow for advanced forms of string interpolation by evaluating the tag as a function to call, passing the template literal elements as arguments.</p>

```js
const foo = function() {
  return 'foo';
}

`bar`; // Noncompliant: `bar` passed as a parameter to function. foo is a string, not a function
```
<p>Therefore, the rule also verifies that template literals don’t start on a separate line.</p>

```js
function foo() {  // <-- Use a function declaration
  return 'foo';
}

`bar`;
```
<p>Or</p>

```js
const foo = function() {
  return 'foo';
}`bar`; // <-- start template literal on same line
```
<h2>Resouces</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Lexical_grammar#automatic_semicolon_insertion">Automatic semicolon
  insertion</a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Grouping#grouping_operator_and_automatic_semicolon_insertion">Grouping operator and automatic semicolon insertion</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Template_literals#tagged_templates">Tagged
  templates</a> </li>
</ul>