<h2>Why is this an issue?</h2>
<p>Exception handling provides a way to help developers handle and recover from runtime errors and unexpected events during program execution. When an
error occurs, an exception is thrown and can be caught by an appropriate try-catch block, allowing the program to handle the exception and prevent the
program from crashing.</p>
<p>A problem arises, when an exception or error class — an instance of <code>Throwable</code> — in a catch clause does not exist, as PHP won’t
generate an error.</p>
<p>This typically occurs when being in a namespace trying to catch PHP built-in exception classes without escaping to the global namespace or
importing the classes.</p>
<p>In summary, this rule raises an issue when, being in a namespace, an undefined class belonging to that namespace is caught.</p>
<h3>What is the potential impact?</h3>
<p>Initially, since the exception is not being caught, it will cause a runtime error in production.</p>
<p>Additionally, this can lead to complex debugging problems as the root cause of the issue may not be obvious.</p>
<h2>How to fix it in Core PHP</h2>
<p>Escape the caught exception to the global namespace or import the class.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```php
namespace Foo\Bar;

try {
    doSomething();
} catch (Exception $e) { // Noncompliant - Exception will never be caught because the class Exception does not exist in the namespace
    echo $e->message;
}
```
<h4>Compliant solution</h4>

```php
namespace Foo\Bar;

try {
    doSomething();
} catch (\Exception $e) { // Compliant: Used by global namespace
    echo $e->message;
}
```

```php
namespace Foo\Bar;

use Exception;

try {
    doSomething();
} catch (Exception $e) { // Compliant: Imported by use statement
    echo $e->message;
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://www.php.net/manual/en/language.exceptions.php">PHP Manual - Exceptions</a> </li>
</ul>