<h2>Why is this an issue?</h2>
<p>In JavaScript, some <code>Array</code> methods do not mutate the existing array that the method was called on, but instead return a new array.
Other methods mutate the array, and their return value differs depending on the method.</p>
<p><code>reverse</code> and <code>sort</code> are mutating methods and, in addition, return the altered version. This rule raises an issue when the
return values of these methods are assigned, which could lead maintainers to overlook the fact that the original array has been modified.</p>

```ts
const reversed = a.reverse(); // Noncompliant: mutating method, no need to assign return value
const sorted = b.sort(); // Noncompliant: mutating method, no need to assign return value
```
<p>Remove the assignment, so that the intent of mutating the original array is clear.</p>

```ts
a.reverse();
b.sort();
```
<p>Or use non-mutating alternatives <code>toReversed</code> and <code>toSorted</code>.</p>

```ts
const reversed = a.toReversed();
const sorted = b.toSorted();
```
<p>Alternatively, change a mutating method into a non-mutating alternative using the spread syntax (<code>…​</code>).</p>

```ts
const reversed = [...a].reverse();
const sorted = [...b].sort();
```
<p>Or <code>slice()</code> to create a copy first.</p>

```ts
const reversed = a.slice().reverse();
const sorted = b.slice().sort();
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array#copying_methods_and_mutating_methods">Array copying
  methods and mutating methods</a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/reverse">Array.prototype.reverse()</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/sort">Array.prototype.sort()</a>
  </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/toReversed">Array.prototype.toReversed()</a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/toSorted">Array.prototype.toSorted()</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Spread_syntax">Spread syntax
  (<code>...</code>)</a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/slice">Array.prototype.slice()</a> </li>
</ul>