An `URL` object that is created and later used without checking the parameters, is possibly vulnerable to a Server-Side-Request-Forgery-Attack (SSRF).
# What Does This Check Look For?
A finding is created, if a vulnarable method is called on an `URL` object, which was initiated with an unsafe parameter.
You can avoid these findings, by checking the properties of the `URL` object before retrieving data with it.
The following cases would be flagged:
```java
new URL(String url).openConnection()
new URL(String url).openStream()
new URL(String url).getContent()

Url someUrl = new URL(String url)
someUrl.vulnerableMethod()
```

# Why is This a Problem?
These code snippets present a potential vulnerability to an SSRF attack, which can be exploited by malicious users.
By purposefully inserting a forged URL, an attacker can trick the program into making unintended calls.
If the URL is not meticulously validated, it opens up the possibility for the attacker to gain unauthorized access to sensitive data.
In a typical SSRF attack, the attacker may manipulate the server to establish connections with internal-only services within the organization's infrastructure or may coerce the server into connecting with arbitrary external systems.

## Exceptions
URLs only need to be checked if they are provided by non-safe sources.
Therefore, URLs that are only provided by safe sources, for example internal developers, don't pose any problems.

# How Can I Resolve This?
1. Don't accept URLs from users.
2. Accept a destination key and use it to look up the target destination associated with the key.
3. Allow complete URLs or validate that the beginning of the URL is part of an allowlist.
4. Check the properties of the created `URL` object before its use (like the host or the used protocol).

## Example (Before)
```java
class DataFetcher{
	
	InputStream getInputStream(String urlString) throws IOException {
		return new URL(urlString).openStream();
	}
}
```

## Example (After)
If you implement the changes number 3 and number 4 from the above-mentioned list, the code could look like this:
```java
class DataFetcher {

	private static final List<String> ALLOWLIST_HOSTS = List.of("www.teamscale.com");

	InputStream getInputStream(String urlString) throws IOException {
		URL url = new URL(location);
		
		if (ALLOWLIST_HOSTS.contains(url.getHost()) && url.getProtocol().equals("https")) {
			return url.openStream();
		}
		
		throw new IOException(urlString + " is not on the allowlist.");
	}
}
```

# Where Can I Learn More?
- [OWASP: Server Side Request Forgery](https://owasp.org/www-community/attacks/Server_Side_Request_Forgery)
- [OWASP Cheat Sheet Series: Server-Side Request Forgery Prevention Cheat Sheet](https://cheatsheetseries.owasp.org/cheatsheets/Server_Side_Request_Forgery_Prevention_Cheat_Sheet.html)
