Using certain discouraged methods can introduce unwanted side effects or reduce performance.
Refactor your code to remove or replace these discouraged method calls with recommended alternatives.

# What Does This Check Look For?

This check identifies usage of discouraged methods in Java code, such as console or debugging methods, that are known to potentially cause performance issues or unintended side effects.

# Why is This a Problem?

Using certain methods in production code can lead to performance degradation or unpredictable behavior.
For example, debugging methods might be left in the code accidentally, which can expose sensitive information or consume unnecessary system resources.
Similarly, methods that manipulate the console might not work as expected in different environments, leading to inconsistent behavior.

# How Can I Resolve This?

- If the method is used for debugging, ensure it is removed or replaced with a more formal logging approach that can be easily controlled in production.
- If the method has a more efficient or safer alternative, refactor the code to use the recommended approach.
- Assess the necessity of the method call and remove it if it serves no critical purpose.

# Examples

## Non-Compliant

```java
public void processData() {
    System.out.println("Processing data"); // non-compliant: using console logging
}
```

## Compliant

```java
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataProcessor {
    private static final Logger logger = LoggerFactory.getLogger(DataProcessor.class);
    
    public void processData() {
        logger.debug("Processing data"); // compliant: using a logging framework
    }
}
```

# Where Can I Learn More?

- [Guideline: OWASP Logging Cheat Sheet](https://cheatsheetseries.owasp.org/cheatsheets/Logging_Cheat_Sheet.html)
- [Introduction to Java Logging](https://www.baeldung.com/java-logging-intro)