
Preprocessor macros that are expanded often can result in a blow-up of (binary) code size.
To avoid this, macros can sometimes be converted to functions that are evaluated at runtime.

# What Does This Check Look For?
This check searches for macro definitions that are expanded often and are replaced with many tokens (as an indicator of complex code leading to binary-size increase).
This check ignores macro definitions that are simply removed, expanded by only one token, or expanded by a constant expression, as there is no binary-size cost in these cases.

# Why is This a Problem?
Macros that are used often and are expanded with a lot of code look like a neat solution to hide a lot of complexity behind a simple "Macro call".
But since they are evaluated at compile time, they can actually lead to a lot of code being compiled and stored in the binary program.
To keep the size of the binary small, using a global function is sometimes better. When using C++11 or later, consider the usage of `constexpr` functions evaluated at compile-time.
Besides type-safety, the usage of `constexpr` functions in shared libraries may further reduce the binary size, see Sandor Dargo's blog entry linked below.

# How Can I Resolve This?
If the macro does not need to be resolved at compile time, replace it with a function call.

# Examples
## Non-Compliant
```C++
#define DIV_BY_42(V)  return V / 42;
```
## Compliant (C++)
```C++
constexpr int divBy42(int v) {
    return v / 42;
}
```
## Compliant (C)
```C
inline int divBy42(int v) {
    return v / 42;
}
```

# Note
The finding properties contain an "average operation count of the expansions".
This counts how many operations are added by the expansions of the macro, on average.
As "operation", we count arithmetic operations (+,-, ...), function calls, comparisons (==, >, <, ...) and assignments (==, ...).

# Where Can I Learn More?
- [Sandor Dargo's blog entry: Constexpr functions for smaller binary size?](https://www.sandordargo.com/blog/2023/09/13/constexpr-and-binary-sizes)