Having too many constructors can make a class difficult to understand and use efficiently.
Consider simplifying the class by reducing the number of constructors.

# What Does This Check Look For?

This check reports a finding when a class contains more than the configured number of constructors, which by default is set to 5.

# Why is This a Problem?

A class with too many constructors may lead to confusion about which constructor should be used for object instantiation in different contexts.
It may also suggest that the class has too many responsibilities, violating the Single Responsibility Principle.
Maintaining and extending such a class can become challenging, as each constructor modification might impact many parts of the codebase.

# How Can I Resolve This?

Reducing the number of constructors can often be achieved by adopting alternative patterns such as the builder pattern, static factory methods, or simply combining similar constructors.
Consider whether some of the constructors can be replaced with static factory methods that return an instance of the class. These methods can have meaningful names, making the code more readable.
Reevaluate the class responsibilities; splitting the class into smaller, more focused classes might reduce the need for multiple constructors.

# Examples

## Non-Compliant

```java
public class Car {
	private String make;
	private String model;
	private int year;
	private String color;
	private String engineType;

	public Car(String make) {
		this.make = make;
	}

	public Car(String make, String model) {
		this(make);
		this.model = model;
	}

	public Car(String make, String model, int year) {
		this(make, model);
		this.year = year;
	}

	public Car(String make, String model, int year, String color) {
		this(make, model, year);
		this.color = color;
	}

	public Car(String make, String model, int year, String color, String engineType) {
		this(make, model, year, color);
		this.engineType = engineType;
	}

	public Car(String make, int year, String color, String engineType) {
		this(make, null, year, color, engineType);
	}
}
```

## Compliant

```java
public class Car {
	private final String make;
	private final String model;
	private final int year;
	private final String color;
	private final String engineType;

	private Car(Builder builder) {
		make = builder.make;
		model = builder.model;
		year = builder.year;
		color = builder.color;
		engineType = builder.engineType;
	}

	public static class Builder {
		private final String make;
		private String model;
		private int year;
		private String color;
		private String engineType;

		public Builder(String make) {
			this.make = make;
		}

		public Builder setModel(String model) {
			this.model = model;
			return this;
		}

		public Builder setYear(int year) {
			this.year = year;
			return this;
		}

		public Builder setColor(String color) {
			this.color = color;
			return this;
		}

		public Builder setEngineType(String engineType) {
			this.engineType = engineType;
			return this;
		}

		public Car build() {
			return new Car(this);
		}
	}
}
```
