<h2>Why is this an issue?</h2>
<p>When you use a <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/using"><code>using</code> statement</a>, the
goal is to ensure the correct disposal of an <a href="https://learn.microsoft.com/en-us/dotnet/api/system.idisposable"><code>IDisposable</code></a>
instance when the control leaves the <code>using</code> statement block.</p>
<p>If you return that <code>IDisposable</code> instance inside the block, <code>using</code> will dispose it before the caller can use it, likely
causing exceptions at runtime. You should either remove <code>using</code> statement or avoid returning the <code>IDisposable</code> in the
<code>using</code> statement block.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
public FileStream WriteToFile(string path, string text)
{
  using (var fs = File.Create(path)) // Noncompliant: 'fs' is disposed at the end of the using scope
  {
    var bytes = Encoding.UTF8.GetBytes(text);
    fs.Write(bytes, 0, bytes.Length);
    return fs;
  }
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
public FileStream WriteToFile(string path, string text)
{
  var fs = File.Create(path);
  var bytes = Encoding.UTF8.GetBytes(text);
  fs.Write(bytes, 0, bytes.Length);
  return fs; // Compliant: 'fs' is not disposed once the end of the scope is reached and the caller can use it
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/using">using statement - ensure the
  correct use of disposable objects</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/api/system.idisposable">IDisposable</a> </li>
</ul>

