The comparison of `Optional` instances to `null` is an inappropriate use of the standard library
and indicates potential API design issues.

Instead, it is recommended to utilize the rich functions provided by `Optional`, such as `orElse(T other)`.

# What Does This Check Look For?

This check detects comparisons of `Optional` instances and `null`.

# Why is This a Problem?

The purpose of `Optional` is to **explicitly represent the possibility of a value's absence**.
This feature is offering a more expressive and type-safe alternative to using `null`.
`Optional` is primarily intended as a return type for methods where the absence of a value needs to be conveyed.
While a variable of type `Optional` should never be `null` itself, it can represent a `null` value.
When `Optional` instances are compared to `null`, it undermines the intended use of `Optional`,
leading to less readable code and increased error susceptibility.

# How Can I Resolve This?

The resolution depends on the intent of the `null` comparison:
1. **Intent**: Check for the absence of the value.  
   **Resolve**: Use `isPresent()` if you don't need the value afterward. 
   Otherwise, safely unwrap the value as demonstrated in the examples.
2. **Design issue**: The `Optional` variable itself can be `null`.  
   **Resolve**: This situation should never occur.
   Reconsider your design choices in order to align them with established coding conventions or guidelines.

# Examples
## Non-Compliant

```java
Optional<Integer> optionalValue = Optional.of(1337);

// original intention: print value only if it is present
if (optionalValue != null) { // This causes the check to report a finding
    // [!] Caution: The get() function can throw a NoSuchElementException because it is an unsafe call
    System.out.println("Optional value: " + optionalValue.get());
}
```

## Compliant

```java
Optional<Integer> optionalValue = Optional.of(1337);

// print value only if optional value is present (as lambda)
optionalValue.ifPresent(opV -> System.out.println("Optional value: " + opV));

// print the value of the optional or invoke external function to receive different value
System.out.println("Optional value: " + optionalValue.orElseGet(Example::computeValue));

// print value if present or another (2) if not present
System.out.println("Optional value: " + optionalValue.orElse(69));
```

# Where Can I Learn More?
- [Baeldung Optional Course](https://www.baeldung.com/java-optional)
- [Java Optional Documentation](https://docs.oracle.com/en/java/javase/11/docs/api/java.base/java/util/Optional.html)