<h2>Why is this an issue?</h2>
<p>Using <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/arrays/multidimensional-arrays">multidimensional</a> and <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/arrays/jagged-arrays">jagged</a> arrays as method parameters in C# can be
challenging for developers.</p>
<p>When these methods are exposed to external users, it requires advanced language knowledge for effective usage.</p>
<p>Determining the appropriate data to pass to these parameters may not be intuitive.</p>
<pre data-diff-id="1" data-diff-type="noncompliant">
public class Program
{
    public void WriteMatrix(int[][] matrix) // Noncompliant: data type is not intuitive
    {
        // ...
    }
}
</pre>
<p>In this example, it cannot be inferred easily what the matrix should look like. Is it a 2x2 Matrix or even a triangular Matrix?</p>
<p>Using a collection, data structure, or class that provides a more suitable representation of the required data is recommended instead of a
multidimensional array or jagged array to enhance code readability.</p>
<pre data-diff-id="1" data-diff-type="compliant">
public class Matrix2x2
{
    // ...
}

public class Program
{
    public void WriteMatrix(Matrix2x2 matrix) // Compliant: data type is intuitive
    {
        // ...
    }
}
</pre>
<p>As a result, avoiding exposing such methods to external users is recommended.</p>
<h3>Exceptions</h3>
<p>However, using multidimensional and jagged array method parameters internally, such as in <code>private</code> or <code>internal</code> methods or
within <code>internal</code> classes, is compliant since they are not publicly exposed.</p>
<pre>
public class FirstClass
{
    private void UpdateMatrix(int[][] matrix) // Compliant: method is private
    {
        // ...
    }
}

internal class SecondClass
{
    public void UpdateMatrix(int[][] matrix) // Compliant: class is internal
    {
        // ...
    }
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/standard/collections/">Collections and Data Structures</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/arrays/jagged-arrays">Jagged Arrays</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/arrays/multidimensional-arrays">Multidimensional Arrays</a> </li>
</ul>

