<h2>Why is this an issue?</h2>
<p>In JavaScript, <code>null</code> and <code>undefined</code> are primitive values that do not have properties or methods. When accessing a property
on a <code>null</code> or <code>undefined</code> value, JavaScript tries to access the property of an object that does not exist, which results in a
<code>TypeError</code>.</p>
<p>This can cause the program to crash or behave unexpectedly, which can be difficult to debug.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```js
let foo = null;
console.log(foo.bar); // Noncompliant: TypeError will be thrown
```
<h4>Compliant solution</h4>
<p>The simplest solution is to check in a <code>if</code> condition the equality to <code>null</code> or <code>undefined</code>. With the <a
href="https://262.ecma-international.org/6.0/#sec-abstract-equality-comparison">abstract equality operator</a> it is not required to check both, as
these operators consider <code>null</code> and <code>undefined</code> to be equals.</p>

```js
let foo;
if (foo != null) {
  console.log(foo.bar);
}
```
<p>Also, the logical AND operator (<code>&amp;&amp;</code>) can be used to check if a variable is truthy before attempting to access its properties.
The expression will short-circuit and return the falsy value instead of attempting to access its properties.</p>

```js
let foo = null;
console.log(foo && foo.bar);
```
<p>Alternatively, use the optional chaining operator (<code>?.</code>) to check if the variable is <code>null</code> or <code>undefined</code> before
attempting to access its property. This operator is more readable and concise, especially when dealing with nested properties.</p>

```js
let foo = null;
console.log(foo?.bar);
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/undefined"><code>undefined</code></a>
  </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/null"><code>null</code></a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/TypeError"><code>TypeError</code></a>
  </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Errors/No_properties">TypeError: "x" has no
  properties</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Optional_chaining">Optional chaining
  (<code>?.</code>)</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Operators/Logical_AND">Logical AND
  (<code>&amp;&amp;</code>)</a> </li>
</ul>