<h2>Why is this an issue?</h2>
<p>Getters and setters provide a way to enforce encapsulation by providing methods that give controlled access to class fields. However, in classes
with multiple fields, it is not unusual that copy and paste is used to quickly create the needed getters and setters, which can result in the wrong
field being accessed by a getter or setter.</p>
<p>This rule raises an issue in the following cases:</p>
<ul>
  <li> A setter does not update the field with the corresponding name (if it exists). </li>
  <li> A getter:
    <ul>
      <li> does not return any value </li>
      <li> does not access the field with the corresponding name (if it exists). </li>
    </ul>  </li>
</ul>
<p>Underscore prefixes for fields are supported, so <code>setX()</code> can assign a value to <code>_x</code>.</p>
<p>The following type of getters and setters are supported:</p>
<ul>
  <li> <code>getX()</code> and <code>setX()</code> </li>
</ul>

```js
class A {
  #y: number = 0;
  setY(val: number) { // Noncompliant: field '#y' is not updated
  }
}
```

```js
class A {
  #y: number = 0;
  setY(val: number) {
    this.#y = val;
  }
}
```
<ul>
  <li> <code>get x()</code> and <code>set x()</code> </li>
</ul>

```js
class A {
  _x: number = 0;
  #y: number = 0;

  get x() { // Noncompliant: field '_x' is not used in the return value
    return this.#y;
  }

  get y() { // Noncompliant: method may not return any value
    if (condition) {
      return #y;
    }
  }
}
```

```js
class A {
  _x: number = 0;
  #y: number = 0;

  get x() {
    return this._x;
  }
  get y() {
    if (condition) {
      return #y;
    }
    return 1;
  }
}
```
<ul>
  <li> getters and setters defined with <code>Object.defineProperty()</code> </li>
</ul>

```js
let x = 0;
let y = 0;
Object.defineProperty(o, 'x', {
  get() { // Noncompliant: variable 'x' is not used in the return value
    return y;
  }
});
```

```js
let x = 0;
let y = 0;
Object.defineProperty(o, 'x', {
  get() {
    return x;
  }
});
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Functions/get">get</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Functions/set">set</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Classes/Private_class_fields">Private class
  features</a> </li>
</ul>