<p>When using <code>SecureRandom</code>, it is important not to use predictable seeds. This class is used to generate cryptographically strong random
numbers. Using a predictable seed will make its output predictable as well, which counteracts the use case of <code>SecureRandom</code>.</p>
<h2>Why is this an issue?</h2>
<p><code>java.security.SecureRandom</code> is often used to generate random values for cryptographic algorithms. When a random number generator is
used for cryptographic purposes, the generated numbers must be as random and unpredictable as possible. When <code>SecureRandom</code> is improperly
seeded with a constant or a predictable value, its output will also be predictable.</p>
<p>This can have severe security implications for cryptographic operations that rely on the randomness of the generated numbers. By using a
predictable seed, an attacker can potentially guess or deduce the generated numbers, compromising the security of whatever cryptographic algorithm
relies on <code>SecureRandom</code>.</p>
<h3>What is the potential impact?</h3>
<p>It is crucial to understand that the strength of cryptographic algorithms heavily relies on the quality of the random numbers used. By improperly
seeding the <code>SecureRandom</code> class, we introduce a significant weakness that can be exploited by attackers.</p>
<h4>Insecure cryptographic keys</h4>
<p>One of the primary use cases for the <code>SecureRandom</code> class is generating cryptographic keys. If an attacker can predict the seed used to
initialize the SecureRandom instance, they may be able to derive the same keys. Depending on the use case, this can lead to multiple severe outcomes,
such as:</p>
<ul>
  <li> Being able to decrypt sensitive documents, leading to privacy breaches or identity theft. </li>
  <li> Gaining access to a private key used for signing, allowing an attacker to forge digital signatures and impersonate legitimate entities. </li>
  <li> Bypassing authentication mechanisms that rely on public-key infrastructure (PKI), which can be abused to gain unauthorized access to systems or
  networks. </li>
</ul>
<h4>Session hijacking and man-in-the-middle attack</h4>
<p>Another scenario where this vulnerability can be exploited is in the generation of session tokens or nonces for secure communication protocols. If
an attacker can predict the seed used to generate these tokens, they can impersonate legitimate users or intercept sensitive information.</p>
<h2>How to fix it in Java SE</h2>
<h3>Code examples</h3>
<p>The following code uses a cryptographically strong random number generator to generate data that is not cryptographically strong.</p>
<h4>Noncompliant code example</h4>

```kotlin
import java.security.SecureRandom

val sr = SecureRandom()
sr.setSeed(123456L) // Noncompliant
val v = sr.nextInt()
```

```kotlin
import java.security.SecureRandom

val sr = SecureRandom("abcdefghijklmnop".toByteArray(charset("us-ascii"))) // Noncompliant
val v = sr.nextInt()
```
<h4>Compliant solution</h4>

```kotlin
import java.security.SecureRandom

val sr = SecureRandom()
val v = sr.nextInt()
```
<p>This solution is available for JDK 1.8 and higher.</p>

```kotlin
import java.security.SecureRandom

val sr = SecureRandom.getInstanceStrong()
val v = sr.nextInt()
```
<h3>How does this work?</h3>
<p>When the randomly generated data needs to be cryptographically strong, <code>SecureRandom</code> is the correct class to use. However, its
documentation also cites that "any seed material passed to a <code>SecureRandom</code> object must be unpredictable". When no seed is passed by the
user to the object, the <code>SecureRandom</code> object chooses an unpredictable seed by default. Therefore, the easiest way to fix the issue is to
use the default constructor without any calls to <code>SecureObject.setSeed()</code>.</p>
<p>To go the extra mile, <code>SecureObject.getInstanceStrong()</code> returns an instance of <code>SecureObject</code> that is guaranteed to use a
strong algorithm for its number generation.</p>
<p>If the randomly generated data is not used for cryptographic purposes and is not business critical, it may be a better choice to use
<code>java.util.Random</code> instead. In this case, setting a predictable seed may be acceptable depending on the situation.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Java Documentation - <a href="https://docs.oracle.com/en/java/javase/17/docs/api/java.base/java/security/SecureRandom.html">Class
  <code>java.security.SecureRandom</code></a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">OWASP Top 10 2021 Category A2</a> - Cryptographic Failures </li>
  <li> <a href="https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration">OWASP Top 10 2017 Category A6</a> - Security
  Misconfiguration </li>
  <li> <a href="https://cwe.mitre.org/data/definitions/330">MITRE, CWE-330</a> - Use of Insufficiently Random Values </li>
  <li> <a href="https://cwe.mitre.org/data/definitions/332">MITRE, CWE-332</a> - Insufficient Entropy in PRNG </li>
  <li> <a href="https://cwe.mitre.org/data/definitions/336">MITRE, CWE-336</a> - Same Seed in Pseudo-Random Number Generator (PRNG) </li>
  <li> <a href="https://cwe.mitre.org/data/definitions/337">MITRE, CWE-337</a> - Predictable Seed in Pseudo-Random Number Generator (PRNG) </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/display/java/MSC63-J.+Ensure+that+SecureRandom+is+properly+seeded">CERT, MSC63J.</a> - Ensure that
  SecureRandom is properly seeded </li>
</ul>