import type { TSESLint } from '@typescript-eslint/utils';
/**
 * When leveraging the convertAnnotatedSourceToFailureCase() utility, the
 * following characters are eligible to be used in the source code of expected
 * failure cases within ESLint unit tests in order to provide an easy way to
 * annotate where one or more ESLint errors are expected to occur within that
 * source.
 *
 * See the convertAnnotatedSourceToFailureCase() utility itself for more details.
 */
export declare const SPECIAL_UNDERLINE_CHARS: readonly ["~", "^", "#", "%", "¶", "*", "¨", "@"];
type MultipleErrorOptions<TMessageIds extends string> = BaseErrorOptions & {
    readonly messages: readonly (Message<TMessageIds> & {
        readonly char: (typeof SPECIAL_UNDERLINE_CHARS)[number];
    })[];
};
type BaseErrorOptions = {
    readonly description: string;
    readonly annotatedSource: string;
    readonly options?: readonly unknown[];
    readonly annotatedOutput?: string;
    readonly filename?: string;
    readonly only?: boolean;
};
type Message<TMessageIds extends string> = {
    readonly messageId: TMessageIds;
    readonly data?: Record<string, unknown>;
    readonly suggestions?: TSESLint.SuggestionOutput<TMessageIds>[];
};
type SingleErrorOptions<TMessageIds extends string> = BaseErrorOptions & Message<TMessageIds>;
/**
 * convertAnnotatedSourceToFailureCase() provides an ergonomic way to easily write
 * expected failure cases for ESLint rules by allowing you to directly annotate the
 * source code for the case with one or more of the values in `SPECIAL_UNDERLINE_CHARS`.
 *
 * This not only makes the unit tests easier to write because of the time saved in figuring
 * out location data in terms of lines and columns, but also far easier to read, which is
 * arguably much more important.
 *
 * Here is a real-world example of using the utility:
 *
 * ```ts
 *  convertAnnotatedSourceToFailureCase({
 *    description: 'should fail when Pipe has no prefix ng',
 *    annotatedSource: `
 *        @Pipe({
 *          name: 'foo-bar'
 *                ~~~~~~~~~
 *        })
 *        class Test {}
 *    `,
 *    messageId: 'pipePrefix,
 *    options: [{ prefixes: ['ng'] }],
 *    data: { prefixes: '"ng"' },
 *  }),
 * ```
 */
export declare function convertAnnotatedSourceToFailureCase<TMessageIds extends string>(errorOptions: SingleErrorOptions<TMessageIds>): TSESLint.InvalidTestCase<TMessageIds, readonly unknown[]>;
export declare function convertAnnotatedSourceToFailureCase<TMessageIds extends string>(errorOptions: MultipleErrorOptions<TMessageIds>): TSESLint.InvalidTestCase<TMessageIds, readonly unknown[]>;
export {};
